"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _reactNative = require("react-native");

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

function convertErrors(errs) {
  if (!errs) {
    return null;
  }

  return Array.isArray(errs) ? errs.filter(e => !!e) : [errs];
}

function checkValidInput(usedKey, value) {
  const isValuePassed = arguments.length > 1;

  if (typeof usedKey !== 'string') {
    console.warn(`[AsyncStorageLegacy] Using "${typeof usedKey}" type for key is not supported. This can lead to unexpected behavior/errors. Use string instead.\nKey passed: ${usedKey}\n`);
  }

  if (isValuePassed && typeof value !== 'string') {
    if (value == null) {
      throw new Error(`[AsyncStorageLegacy] Passing "null" or "undefined" as value is not supported. If you need to remove a value, use ".removeSingle" method instead.\nUsed key: ${usedKey}\n`);
    } else {
      console.warn(`[AsyncStorageLegacy] The value for key "${usedKey}" is not a string. This can lead to unexpected behavior/errors. Consider stringifying it.\nPassed value: ${value}\nPassed key: ${usedKey}\n`);
    }
  }
}

class LegacyAsyncStorage {
  constructor() {
    _defineProperty(this, "_asyncStorageNativeModule", void 0);

    this._asyncStorageNativeModule = _reactNative.NativeModules.RNC_AsyncSQLiteDBStorage || _reactNative.NativeModules.RNCAsyncStorage;

    if (!this._asyncStorageNativeModule) {
      throw new Error('[@RNC/AsyncStorage]: NativeModule: AsyncStorage is null.');
    }
  }

  async getSingle(key, _) {
    checkValidInput(key);
    return new Promise((resolve, reject) => {
      this._asyncStorageNativeModule.multiGet([key], function (errors, result) {
        const value = result && result[0] && result[0][1] || null;
        const errs = convertErrors(errors);

        if (errs && errs.length) {
          reject(errs[0]);
        } else {
          resolve(value);
        }
      });
    });
  }

  async setSingle(key, value, _) {
    checkValidInput(key, value);
    return new Promise((resolve, reject) => {
      this._asyncStorageNativeModule.multiSet([[key, value]], function (errors) {
        const errs = convertErrors(errors);

        if (errs && errs.length) {
          reject(errs[0]);
        } else {
          resolve();
        }
      });
    });
  }

  async getMany(keys, _) {
    keys.forEach(k => checkValidInput(k));
    return new Promise((resolve, reject) => {
      this._asyncStorageNativeModule.multiGet(keys, function (errors, result) {
        const value = result.reduce((acc, current) => {
          const key = current[0];
          const val = current[1] || null;
          return _objectSpread({}, acc, {
            [key]: val
          });
        }, {});
        const errs = convertErrors(errors);

        if (errs && errs.length) {
          reject(errs[0]);
        } else {
          resolve(value);
        }
      });
    });
  }

  async setMany(values, _) {
    values.forEach(keyValue => {
      Object.keys(keyValue).forEach(key => {
        checkValidInput(key, keyValue[key]);
      });
    });
    return new Promise((resolve, reject) => {
      const valuesArray = values.map(entry => {
        return [Object.keys(entry)[0], entry];
      });

      this._asyncStorageNativeModule.multiSet([valuesArray], function (errors) {
        const errs = convertErrors(errors);

        if (errs && errs.length) {
          reject(errs[0]);
        } else {
          resolve();
        }
      });
    });
  }

  async removeSingle(key, _) {
    checkValidInput(key);
    return new Promise((resolve, reject) => {
      this._asyncStorageNativeModule.multiRemove([key], function (errors) {
        const errs = convertErrors(errors);

        if (errs && errs.length) {
          reject(errs[0]);
        } else {
          resolve();
        }
      });
    });
  }

  async removeMany(keys, _) {
    keys.forEach(k => checkValidInput(k));
    return new Promise((resolve, reject) => {
      this._asyncStorageNativeModule.multiRemove(keys, function (errors) {
        const errs = convertErrors(errors);

        if (errs && errs.length) {
          reject(errs[0]);
        } else {
          resolve();
        }
      });
    });
  }

  async getKeys(_) {
    return new Promise((resolve, reject) => {
      this._asyncStorageNativeModule.getAllKeys(function (errors, keys) {
        const err = convertErrors(errors);

        if (err && err.length) {
          reject(err[0]);
        } else {
          resolve(keys);
        }
      });
    });
  }

  async dropStorage(_) {
    return new Promise((resolve, reject) => {
      this._asyncStorageNativeModule.clear(function (error) {
        const err = convertErrors(Array.isArray(error) ? error : [error]);

        if (err) {
          reject(err);
        } else {
          resolve();
        }
      });
    });
  }

}

exports.default = LegacyAsyncStorage;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uL3NyYy9pbmRleC50cyJdLCJuYW1lcyI6WyJjb252ZXJ0RXJyb3JzIiwiZXJycyIsIkFycmF5IiwiaXNBcnJheSIsImZpbHRlciIsImUiLCJjaGVja1ZhbGlkSW5wdXQiLCJ1c2VkS2V5IiwidmFsdWUiLCJpc1ZhbHVlUGFzc2VkIiwiYXJndW1lbnRzIiwibGVuZ3RoIiwiY29uc29sZSIsIndhcm4iLCJFcnJvciIsIkxlZ2FjeUFzeW5jU3RvcmFnZSIsImNvbnN0cnVjdG9yIiwiX2FzeW5jU3RvcmFnZU5hdGl2ZU1vZHVsZSIsIk5hdGl2ZU1vZHVsZXMiLCJSTkNfQXN5bmNTUUxpdGVEQlN0b3JhZ2UiLCJSTkNBc3luY1N0b3JhZ2UiLCJnZXRTaW5nbGUiLCJrZXkiLCJfIiwiUHJvbWlzZSIsInJlc29sdmUiLCJyZWplY3QiLCJtdWx0aUdldCIsImVycm9ycyIsInJlc3VsdCIsInNldFNpbmdsZSIsIm11bHRpU2V0IiwiZ2V0TWFueSIsImtleXMiLCJmb3JFYWNoIiwiayIsInJlZHVjZSIsImFjYyIsImN1cnJlbnQiLCJ2YWwiLCJzZXRNYW55IiwidmFsdWVzIiwia2V5VmFsdWUiLCJPYmplY3QiLCJ2YWx1ZXNBcnJheSIsIm1hcCIsImVudHJ5IiwicmVtb3ZlU2luZ2xlIiwibXVsdGlSZW1vdmUiLCJyZW1vdmVNYW55IiwiZ2V0S2V5cyIsImdldEFsbEtleXMiLCJlcnIiLCJkcm9wU3RvcmFnZSIsImNsZWFyIiwiZXJyb3IiXSwibWFwcGluZ3MiOiI7Ozs7Ozs7QUFRQTs7Ozs7O0FBUUEsU0FBU0EsYUFBVCxDQUF1QkMsSUFBdkIsRUFBb0Q7QUFDbEQsTUFBSSxDQUFDQSxJQUFMLEVBQVc7QUFDVCxXQUFPLElBQVA7QUFDRDs7QUFDRCxTQUFPQyxLQUFLLENBQUNDLE9BQU4sQ0FBY0YsSUFBZCxJQUFzQkEsSUFBSSxDQUFDRyxNQUFMLENBQVlDLENBQUMsSUFBSSxDQUFDLENBQUNBLENBQW5CLENBQXRCLEdBQThDLENBQUNKLElBQUQsQ0FBckQ7QUFDRDs7QUFFRCxTQUFTSyxlQUFULENBQXlCQyxPQUF6QixFQUF1Q0MsS0FBdkMsRUFBb0Q7QUFDbEQsUUFBTUMsYUFBYSxHQUFHQyxTQUFTLENBQUNDLE1BQVYsR0FBbUIsQ0FBekM7O0FBRUEsTUFBSSxPQUFPSixPQUFQLEtBQW1CLFFBQXZCLEVBQWlDO0FBQy9CSyxJQUFBQSxPQUFPLENBQUNDLElBQVIsQ0FDRywrQkFBOEIsT0FBT04sT0FBUSxrSEFBaUhBLE9BQVEsSUFEeks7QUFHRDs7QUFFRCxNQUFJRSxhQUFhLElBQUksT0FBT0QsS0FBUCxLQUFpQixRQUF0QyxFQUFnRDtBQUM5QyxRQUFJQSxLQUFLLElBQUksSUFBYixFQUFtQjtBQUNqQixZQUFNLElBQUlNLEtBQUosQ0FDSCwrSkFBOEpQLE9BQVEsSUFEbkssQ0FBTjtBQUdELEtBSkQsTUFJTztBQUNMSyxNQUFBQSxPQUFPLENBQUNDLElBQVIsQ0FDRywyQ0FBMENOLE9BQVEsNEdBQTJHQyxLQUFNLGlCQUFnQkQsT0FBUSxJQUQ5TDtBQUdEO0FBQ0Y7QUFDRjs7QUFFYyxNQUFNUSxrQkFBTixDQUVpQjtBQUc5QkMsRUFBQUEsV0FBVyxHQUFHO0FBQUE7O0FBQ1osU0FBS0MseUJBQUwsR0FDRUMsMkJBQWNDLHdCQUFkLElBQTBDRCwyQkFBY0UsZUFEMUQ7O0FBR0EsUUFBSSxDQUFDLEtBQUtILHlCQUFWLEVBQXFDO0FBQ25DLFlBQU0sSUFBSUgsS0FBSixDQUNKLDBEQURJLENBQU47QUFHRDtBQUNGOztBQUVELFFBQU1PLFNBQU4sQ0FDRUMsR0FERixFQUVFQyxDQUZGLEVBR3dCO0FBQ3RCakIsSUFBQUEsZUFBZSxDQUFDZ0IsR0FBRCxDQUFmO0FBRUEsV0FBTyxJQUFJRSxPQUFKLENBQVksQ0FBQ0MsT0FBRCxFQUFVQyxNQUFWLEtBQXFCO0FBQ3RDLFdBQUtULHlCQUFMLENBQStCVSxRQUEvQixDQUF3QyxDQUFDTCxHQUFELENBQXhDLEVBQStDLFVBQzdDTSxNQUQ2QyxFQUU3Q0MsTUFGNkMsRUFHN0M7QUFDQSxjQUFNckIsS0FBSyxHQUFJcUIsTUFBTSxJQUFJQSxNQUFNLENBQUMsQ0FBRCxDQUFoQixJQUF1QkEsTUFBTSxDQUFDLENBQUQsQ0FBTixDQUFVLENBQVYsQ0FBeEIsSUFBeUMsSUFBdkQ7QUFDQSxjQUFNNUIsSUFBSSxHQUFHRCxhQUFhLENBQUM0QixNQUFELENBQTFCOztBQUNBLFlBQUkzQixJQUFJLElBQUlBLElBQUksQ0FBQ1UsTUFBakIsRUFBeUI7QUFDdkJlLFVBQUFBLE1BQU0sQ0FBQ3pCLElBQUksQ0FBQyxDQUFELENBQUwsQ0FBTjtBQUNELFNBRkQsTUFFTztBQUNMd0IsVUFBQUEsT0FBTyxDQUFDakIsS0FBRCxDQUFQO0FBQ0Q7QUFDRixPQVhEO0FBWUQsS0FiTSxDQUFQO0FBY0Q7O0FBRUQsUUFBTXNCLFNBQU4sQ0FDRVIsR0FERixFQUVFZCxLQUZGLEVBR0VlLENBSEYsRUFJaUI7QUFDZmpCLElBQUFBLGVBQWUsQ0FBQ2dCLEdBQUQsRUFBTWQsS0FBTixDQUFmO0FBRUEsV0FBTyxJQUFJZ0IsT0FBSixDQUFZLENBQUNDLE9BQUQsRUFBVUMsTUFBVixLQUFxQjtBQUN0QyxXQUFLVCx5QkFBTCxDQUErQmMsUUFBL0IsQ0FBd0MsQ0FBQyxDQUFDVCxHQUFELEVBQU1kLEtBQU4sQ0FBRCxDQUF4QyxFQUF3RCxVQUN0RG9CLE1BRHNELEVBRXREO0FBQ0EsY0FBTTNCLElBQUksR0FBR0QsYUFBYSxDQUFDNEIsTUFBRCxDQUExQjs7QUFDQSxZQUFJM0IsSUFBSSxJQUFJQSxJQUFJLENBQUNVLE1BQWpCLEVBQXlCO0FBQ3ZCZSxVQUFBQSxNQUFNLENBQUN6QixJQUFJLENBQUMsQ0FBRCxDQUFMLENBQU47QUFDRCxTQUZELE1BRU87QUFDTHdCLFVBQUFBLE9BQU87QUFDUjtBQUNGLE9BVEQ7QUFVRCxLQVhNLENBQVA7QUFZRDs7QUFFRCxRQUFNTyxPQUFOLENBQ0VDLElBREYsRUFFRVYsQ0FGRixFQUdvQztBQUNsQ1UsSUFBQUEsSUFBSSxDQUFDQyxPQUFMLENBQWFDLENBQUMsSUFBSTdCLGVBQWUsQ0FBQzZCLENBQUQsQ0FBakM7QUFFQSxXQUFPLElBQUlYLE9BQUosQ0FBWSxDQUFDQyxPQUFELEVBQVVDLE1BQVYsS0FBcUI7QUFDdEMsV0FBS1QseUJBQUwsQ0FBK0JVLFFBQS9CLENBQXdDTSxJQUF4QyxFQUE4QyxVQUM1Q0wsTUFENEMsRUFFNUNDLE1BRjRDLEVBRzVDO0FBQ0EsY0FBTXJCLEtBQXVCLEdBQUdxQixNQUFNLENBQUNPLE1BQVAsQ0FDOUIsQ0FBQ0MsR0FBRCxFQUFNQyxPQUFOLEtBQTZCO0FBQzNCLGdCQUFNaEIsR0FBRyxHQUFHZ0IsT0FBTyxDQUFDLENBQUQsQ0FBbkI7QUFDQSxnQkFBTUMsR0FBRyxHQUFHRCxPQUFPLENBQUMsQ0FBRCxDQUFQLElBQWMsSUFBMUI7QUFDQSxtQ0FDS0QsR0FETDtBQUVFLGFBQUNmLEdBQUQsR0FBT2lCO0FBRlQ7QUFJRCxTQVI2QixFQVM5QixFQVQ4QixDQUFoQztBQVdBLGNBQU10QyxJQUFJLEdBQUdELGFBQWEsQ0FBQzRCLE1BQUQsQ0FBMUI7O0FBQ0EsWUFBSTNCLElBQUksSUFBSUEsSUFBSSxDQUFDVSxNQUFqQixFQUF5QjtBQUN2QmUsVUFBQUEsTUFBTSxDQUFDekIsSUFBSSxDQUFDLENBQUQsQ0FBTCxDQUFOO0FBQ0QsU0FGRCxNQUVPO0FBQ0x3QixVQUFBQSxPQUFPLENBQUNqQixLQUFELENBQVA7QUFDRDtBQUNGLE9BckJEO0FBc0JELEtBdkJNLENBQVA7QUF3QkQ7O0FBRUQsUUFBTWdDLE9BQU4sQ0FDRUMsTUFERixFQUVFbEIsQ0FGRixFQUdpQjtBQUNma0IsSUFBQUEsTUFBTSxDQUFDUCxPQUFQLENBQWVRLFFBQVEsSUFBSTtBQUN4QkMsTUFBQUEsTUFBTSxDQUFDVixJQUFQLENBQVlTLFFBQVosQ0FBRCxDQUFvQ1IsT0FBcEMsQ0FBNENaLEdBQUcsSUFBSTtBQUNqRGhCLFFBQUFBLGVBQWUsQ0FBQ2dCLEdBQUQsRUFBTW9CLFFBQVEsQ0FBQ3BCLEdBQUQsQ0FBZCxDQUFmO0FBQ0QsT0FGRDtBQUdELEtBSkQ7QUFNQSxXQUFPLElBQUlFLE9BQUosQ0FBWSxDQUFDQyxPQUFELEVBQVVDLE1BQVYsS0FBcUI7QUFDdEMsWUFBTWtCLFdBQVcsR0FBR0gsTUFBTSxDQUFDSSxHQUFQLENBQVdDLEtBQUssSUFBSTtBQUN0QyxlQUFPLENBQUNILE1BQU0sQ0FBQ1YsSUFBUCxDQUFZYSxLQUFaLEVBQW1CLENBQW5CLENBQUQsRUFBNkJBLEtBQTdCLENBQVA7QUFDRCxPQUZtQixDQUFwQjs7QUFHQSxXQUFLN0IseUJBQUwsQ0FBK0JjLFFBQS9CLENBQXdDLENBQUNhLFdBQUQsQ0FBeEMsRUFBdUQsVUFDckRoQixNQURxRCxFQUVyRDtBQUNBLGNBQU0zQixJQUFJLEdBQUdELGFBQWEsQ0FBQzRCLE1BQUQsQ0FBMUI7O0FBQ0EsWUFBSTNCLElBQUksSUFBSUEsSUFBSSxDQUFDVSxNQUFqQixFQUF5QjtBQUN2QmUsVUFBQUEsTUFBTSxDQUFDekIsSUFBSSxDQUFDLENBQUQsQ0FBTCxDQUFOO0FBQ0QsU0FGRCxNQUVPO0FBQ0x3QixVQUFBQSxPQUFPO0FBQ1I7QUFDRixPQVREO0FBVUQsS0FkTSxDQUFQO0FBZUQ7O0FBRUQsUUFBTXNCLFlBQU4sQ0FBbUJ6QixHQUFuQixFQUFpQ0MsQ0FBakMsRUFBb0U7QUFDbEVqQixJQUFBQSxlQUFlLENBQUNnQixHQUFELENBQWY7QUFFQSxXQUFPLElBQUlFLE9BQUosQ0FBa0IsQ0FBQ0MsT0FBRCxFQUFVQyxNQUFWLEtBQXFCO0FBQzVDLFdBQUtULHlCQUFMLENBQStCK0IsV0FBL0IsQ0FBMkMsQ0FBQzFCLEdBQUQsQ0FBM0MsRUFBa0QsVUFDaERNLE1BRGdELEVBRWhEO0FBQ0EsY0FBTTNCLElBQUksR0FBR0QsYUFBYSxDQUFDNEIsTUFBRCxDQUExQjs7QUFDQSxZQUFJM0IsSUFBSSxJQUFJQSxJQUFJLENBQUNVLE1BQWpCLEVBQXlCO0FBQ3ZCZSxVQUFBQSxNQUFNLENBQUN6QixJQUFJLENBQUMsQ0FBRCxDQUFMLENBQU47QUFDRCxTQUZELE1BRU87QUFDTHdCLFVBQUFBLE9BQU87QUFDUjtBQUNGLE9BVEQ7QUFVRCxLQVhNLENBQVA7QUFZRDs7QUFFRCxRQUFNd0IsVUFBTixDQUFpQmhCLElBQWpCLEVBQXVDVixDQUF2QyxFQUEwRTtBQUN4RVUsSUFBQUEsSUFBSSxDQUFDQyxPQUFMLENBQWFDLENBQUMsSUFBSTdCLGVBQWUsQ0FBQzZCLENBQUQsQ0FBakM7QUFFQSxXQUFPLElBQUlYLE9BQUosQ0FBa0IsQ0FBQ0MsT0FBRCxFQUFVQyxNQUFWLEtBQXFCO0FBQzVDLFdBQUtULHlCQUFMLENBQStCK0IsV0FBL0IsQ0FBMkNmLElBQTNDLEVBQWlELFVBQy9DTCxNQUQrQyxFQUUvQztBQUNBLGNBQU0zQixJQUFJLEdBQUdELGFBQWEsQ0FBQzRCLE1BQUQsQ0FBMUI7O0FBQ0EsWUFBSTNCLElBQUksSUFBSUEsSUFBSSxDQUFDVSxNQUFqQixFQUF5QjtBQUN2QmUsVUFBQUEsTUFBTSxDQUFDekIsSUFBSSxDQUFDLENBQUQsQ0FBTCxDQUFOO0FBQ0QsU0FGRCxNQUVPO0FBQ0x3QixVQUFBQSxPQUFPO0FBQ1I7QUFDRixPQVREO0FBVUQsS0FYTSxDQUFQO0FBWUQ7O0FBRUQsUUFBTXlCLE9BQU4sQ0FBYzNCLENBQWQsRUFBMkQ7QUFDekQsV0FBTyxJQUFJQyxPQUFKLENBQVksQ0FBQ0MsT0FBRCxFQUFVQyxNQUFWLEtBQXFCO0FBQ3RDLFdBQUtULHlCQUFMLENBQStCa0MsVUFBL0IsQ0FBMEMsVUFDeEN2QixNQUR3QyxFQUV4Q0ssSUFGd0MsRUFHeEM7QUFDQSxjQUFNbUIsR0FBRyxHQUFHcEQsYUFBYSxDQUFDNEIsTUFBRCxDQUF6Qjs7QUFFQSxZQUFJd0IsR0FBRyxJQUFJQSxHQUFHLENBQUN6QyxNQUFmLEVBQXVCO0FBQ3JCZSxVQUFBQSxNQUFNLENBQUMwQixHQUFHLENBQUMsQ0FBRCxDQUFKLENBQU47QUFDRCxTQUZELE1BRU87QUFDTDNCLFVBQUFBLE9BQU8sQ0FBQ1EsSUFBRCxDQUFQO0FBQ0Q7QUFDRixPQVhEO0FBWUQsS0FiTSxDQUFQO0FBY0Q7O0FBRUQsUUFBTW9CLFdBQU4sQ0FBa0I5QixDQUFsQixFQUFxRDtBQUNuRCxXQUFPLElBQUlDLE9BQUosQ0FBa0IsQ0FBQ0MsT0FBRCxFQUFVQyxNQUFWLEtBQXFCO0FBQzVDLFdBQUtULHlCQUFMLENBQStCcUMsS0FBL0IsQ0FBcUMsVUFBU0MsS0FBVCxFQUE4QjtBQUNqRSxjQUFNSCxHQUFHLEdBQUdwRCxhQUFhLENBQUNFLEtBQUssQ0FBQ0MsT0FBTixDQUFjb0QsS0FBZCxJQUF1QkEsS0FBdkIsR0FBK0IsQ0FBQ0EsS0FBRCxDQUFoQyxDQUF6Qjs7QUFFQSxZQUFJSCxHQUFKLEVBQVM7QUFDUDFCLFVBQUFBLE1BQU0sQ0FBQzBCLEdBQUQsQ0FBTjtBQUNELFNBRkQsTUFFTztBQUNMM0IsVUFBQUEsT0FBTztBQUNSO0FBQ0YsT0FSRDtBQVNELEtBVk0sQ0FBUDtBQVdEOztBQW5MNkIiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIENvcHlyaWdodCAoYykgUmVhY3QgTmF0aXZlIENvbW11bml0eS5cbiAqXG4gKiBUaGlzIHNvdXJjZSBjb2RlIGlzIGxpY2Vuc2VkIHVuZGVyIHRoZSBNSVQgbGljZW5zZSBmb3VuZCBpbiB0aGVcbiAqIExJQ0VOU0UgZmlsZSBpbiB0aGUgcm9vdCBkaXJlY3Rvcnkgb2YgdGhpcyBzb3VyY2UgdHJlZS5cbiAqXG4gKi9cblxuaW1wb3J0IHtOYXRpdmVNb2R1bGVzfSBmcm9tICdyZWFjdC1uYXRpdmUnO1xuaW1wb3J0IHtcbiAgRW1wdHlTdG9yYWdlTW9kZWwsXG4gIElTdG9yYWdlQmFja2VuZCxcbiAgU3RvcmFnZU9wdGlvbnMsXG59IGZyb20gJ0ByZWFjdC1uYXRpdmUtY29tbXVuaXR5L2FzeW5jLXN0b3JhZ2UnO1xuaW1wb3J0IHtJTGVnYWN5TmF0aXZlTW9kdWxlfSBmcm9tICcuLi90eXBlcy9uYXRpdmVNb2R1bGUnO1xuXG5mdW5jdGlvbiBjb252ZXJ0RXJyb3JzKGVycnM/OiBBcnJheTxFcnJvcj4gfCBFcnJvcikge1xuICBpZiAoIWVycnMpIHtcbiAgICByZXR1cm4gbnVsbDtcbiAgfVxuICByZXR1cm4gQXJyYXkuaXNBcnJheShlcnJzKSA/IGVycnMuZmlsdGVyKGUgPT4gISFlKSA6IFtlcnJzXTtcbn1cblxuZnVuY3Rpb24gY2hlY2tWYWxpZElucHV0KHVzZWRLZXk6IGFueSwgdmFsdWU/OiBhbnkpIHtcbiAgY29uc3QgaXNWYWx1ZVBhc3NlZCA9IGFyZ3VtZW50cy5sZW5ndGggPiAxO1xuXG4gIGlmICh0eXBlb2YgdXNlZEtleSAhPT0gJ3N0cmluZycpIHtcbiAgICBjb25zb2xlLndhcm4oXG4gICAgICBgW0FzeW5jU3RvcmFnZUxlZ2FjeV0gVXNpbmcgXCIke3R5cGVvZiB1c2VkS2V5fVwiIHR5cGUgZm9yIGtleSBpcyBub3Qgc3VwcG9ydGVkLiBUaGlzIGNhbiBsZWFkIHRvIHVuZXhwZWN0ZWQgYmVoYXZpb3IvZXJyb3JzLiBVc2Ugc3RyaW5nIGluc3RlYWQuXFxuS2V5IHBhc3NlZDogJHt1c2VkS2V5fVxcbmAsXG4gICAgKTtcbiAgfVxuXG4gIGlmIChpc1ZhbHVlUGFzc2VkICYmIHR5cGVvZiB2YWx1ZSAhPT0gJ3N0cmluZycpIHtcbiAgICBpZiAodmFsdWUgPT0gbnVsbCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBgW0FzeW5jU3RvcmFnZUxlZ2FjeV0gUGFzc2luZyBcIm51bGxcIiBvciBcInVuZGVmaW5lZFwiIGFzIHZhbHVlIGlzIG5vdCBzdXBwb3J0ZWQuIElmIHlvdSBuZWVkIHRvIHJlbW92ZSBhIHZhbHVlLCB1c2UgXCIucmVtb3ZlU2luZ2xlXCIgbWV0aG9kIGluc3RlYWQuXFxuVXNlZCBrZXk6ICR7dXNlZEtleX1cXG5gLFxuICAgICAgKTtcbiAgICB9IGVsc2Uge1xuICAgICAgY29uc29sZS53YXJuKFxuICAgICAgICBgW0FzeW5jU3RvcmFnZUxlZ2FjeV0gVGhlIHZhbHVlIGZvciBrZXkgXCIke3VzZWRLZXl9XCIgaXMgbm90IGEgc3RyaW5nLiBUaGlzIGNhbiBsZWFkIHRvIHVuZXhwZWN0ZWQgYmVoYXZpb3IvZXJyb3JzLiBDb25zaWRlciBzdHJpbmdpZnlpbmcgaXQuXFxuUGFzc2VkIHZhbHVlOiAke3ZhbHVlfVxcblBhc3NlZCBrZXk6ICR7dXNlZEtleX1cXG5gLFxuICAgICAgKTtcbiAgICB9XG4gIH1cbn1cblxuZXhwb3J0IGRlZmF1bHQgY2xhc3MgTGVnYWN5QXN5bmNTdG9yYWdlPFxuICBUIGV4dGVuZHMgRW1wdHlTdG9yYWdlTW9kZWwgPSBFbXB0eVN0b3JhZ2VNb2RlbFxuPiBpbXBsZW1lbnRzIElTdG9yYWdlQmFja2VuZDxUPiB7XG4gIHByaXZhdGUgcmVhZG9ubHkgX2FzeW5jU3RvcmFnZU5hdGl2ZU1vZHVsZTogSUxlZ2FjeU5hdGl2ZU1vZHVsZTtcblxuICBjb25zdHJ1Y3RvcigpIHtcbiAgICB0aGlzLl9hc3luY1N0b3JhZ2VOYXRpdmVNb2R1bGUgPVxuICAgICAgTmF0aXZlTW9kdWxlcy5STkNfQXN5bmNTUUxpdGVEQlN0b3JhZ2UgfHwgTmF0aXZlTW9kdWxlcy5STkNBc3luY1N0b3JhZ2U7XG5cbiAgICBpZiAoIXRoaXMuX2FzeW5jU3RvcmFnZU5hdGl2ZU1vZHVsZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAnW0BSTkMvQXN5bmNTdG9yYWdlXTogTmF0aXZlTW9kdWxlOiBBc3luY1N0b3JhZ2UgaXMgbnVsbC4nLFxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICBhc3luYyBnZXRTaW5nbGU8SyBleHRlbmRzIGtleW9mIFQ+KFxuICAgIGtleTogSyxcbiAgICBfPzogU3RvcmFnZU9wdGlvbnMsXG4gICk6IFByb21pc2U8VFtLXSB8IG51bGw+IHtcbiAgICBjaGVja1ZhbGlkSW5wdXQoa2V5KTtcblxuICAgIHJldHVybiBuZXcgUHJvbWlzZSgocmVzb2x2ZSwgcmVqZWN0KSA9PiB7XG4gICAgICB0aGlzLl9hc3luY1N0b3JhZ2VOYXRpdmVNb2R1bGUubXVsdGlHZXQoW2tleV0sIGZ1bmN0aW9uKFxuICAgICAgICBlcnJvcnM6IEFycmF5PEVycm9yPixcbiAgICAgICAgcmVzdWx0OiBBcnJheTxbYW55LCBUW0tdIHwgbnVsbF0+LFxuICAgICAgKSB7XG4gICAgICAgIGNvbnN0IHZhbHVlID0gKHJlc3VsdCAmJiByZXN1bHRbMF0gJiYgcmVzdWx0WzBdWzFdKSB8fCBudWxsO1xuICAgICAgICBjb25zdCBlcnJzID0gY29udmVydEVycm9ycyhlcnJvcnMpO1xuICAgICAgICBpZiAoZXJycyAmJiBlcnJzLmxlbmd0aCkge1xuICAgICAgICAgIHJlamVjdChlcnJzWzBdKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICByZXNvbHZlKHZhbHVlKTtcbiAgICAgICAgfVxuICAgICAgfSk7XG4gICAgfSk7XG4gIH1cblxuICBhc3luYyBzZXRTaW5nbGU8SyBleHRlbmRzIGtleW9mIFQ+KFxuICAgIGtleTogSyxcbiAgICB2YWx1ZTogVFtLXSxcbiAgICBfPzogU3RvcmFnZU9wdGlvbnMsXG4gICk6IFByb21pc2U8dm9pZD4ge1xuICAgIGNoZWNrVmFsaWRJbnB1dChrZXksIHZhbHVlKTtcblxuICAgIHJldHVybiBuZXcgUHJvbWlzZSgocmVzb2x2ZSwgcmVqZWN0KSA9PiB7XG4gICAgICB0aGlzLl9hc3luY1N0b3JhZ2VOYXRpdmVNb2R1bGUubXVsdGlTZXQoW1trZXksIHZhbHVlXV0sIGZ1bmN0aW9uKFxuICAgICAgICBlcnJvcnM6IEFycmF5PEVycm9yPixcbiAgICAgICkge1xuICAgICAgICBjb25zdCBlcnJzID0gY29udmVydEVycm9ycyhlcnJvcnMpO1xuICAgICAgICBpZiAoZXJycyAmJiBlcnJzLmxlbmd0aCkge1xuICAgICAgICAgIHJlamVjdChlcnJzWzBdKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICByZXNvbHZlKCk7XG4gICAgICAgIH1cbiAgICAgIH0pO1xuICAgIH0pO1xuICB9XG5cbiAgYXN5bmMgZ2V0TWFueTxLIGV4dGVuZHMga2V5b2YgVD4oXG4gICAga2V5czogQXJyYXk8Sz4sXG4gICAgXz86IFN0b3JhZ2VPcHRpb25zLFxuICApOiBQcm9taXNlPHtbayBpbiBLXTogVFtrXSB8IG51bGx9PiB7XG4gICAga2V5cy5mb3JFYWNoKGsgPT4gY2hlY2tWYWxpZElucHV0KGspKTtcblxuICAgIHJldHVybiBuZXcgUHJvbWlzZSgocmVzb2x2ZSwgcmVqZWN0KSA9PiB7XG4gICAgICB0aGlzLl9hc3luY1N0b3JhZ2VOYXRpdmVNb2R1bGUubXVsdGlHZXQoa2V5cywgZnVuY3Rpb24oXG4gICAgICAgIGVycm9yczogQXJyYXk8RXJyb3I+LFxuICAgICAgICByZXN1bHQ6IEFycmF5PFtLLCBUW0tdXT4sXG4gICAgICApIHtcbiAgICAgICAgY29uc3QgdmFsdWU6IHtbayBpbiBLXTogVFtrXX0gPSByZXN1bHQucmVkdWNlPGFueT4oXG4gICAgICAgICAgKGFjYywgY3VycmVudDogW0ssIFRbS11dKSA9PiB7XG4gICAgICAgICAgICBjb25zdCBrZXkgPSBjdXJyZW50WzBdO1xuICAgICAgICAgICAgY29uc3QgdmFsID0gY3VycmVudFsxXSB8fCBudWxsO1xuICAgICAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgICAgLi4uYWNjLFxuICAgICAgICAgICAgICBba2V5XTogdmFsLFxuICAgICAgICAgICAgfTtcbiAgICAgICAgICB9LFxuICAgICAgICAgIHt9LFxuICAgICAgICApO1xuICAgICAgICBjb25zdCBlcnJzID0gY29udmVydEVycm9ycyhlcnJvcnMpO1xuICAgICAgICBpZiAoZXJycyAmJiBlcnJzLmxlbmd0aCkge1xuICAgICAgICAgIHJlamVjdChlcnJzWzBdKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICByZXNvbHZlKHZhbHVlKTtcbiAgICAgICAgfVxuICAgICAgfSk7XG4gICAgfSk7XG4gIH1cblxuICBhc3luYyBzZXRNYW55PEsgZXh0ZW5kcyBrZXlvZiBUPihcbiAgICB2YWx1ZXM6IEFycmF5PFBhcnRpYWw8e1trIGluIEtdOiBUW2tdfT4+LFxuICAgIF8/OiBTdG9yYWdlT3B0aW9ucyxcbiAgKTogUHJvbWlzZTx2b2lkPiB7XG4gICAgdmFsdWVzLmZvckVhY2goa2V5VmFsdWUgPT4ge1xuICAgICAgKE9iamVjdC5rZXlzKGtleVZhbHVlKSBhcyBBcnJheTxLPikuZm9yRWFjaChrZXkgPT4ge1xuICAgICAgICBjaGVja1ZhbGlkSW5wdXQoa2V5LCBrZXlWYWx1ZVtrZXldKTtcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgcmV0dXJuIG5ldyBQcm9taXNlKChyZXNvbHZlLCByZWplY3QpID0+IHtcbiAgICAgIGNvbnN0IHZhbHVlc0FycmF5ID0gdmFsdWVzLm1hcChlbnRyeSA9PiB7XG4gICAgICAgIHJldHVybiBbT2JqZWN0LmtleXMoZW50cnkpWzBdIGFzIEssIGVudHJ5XTtcbiAgICAgIH0pO1xuICAgICAgdGhpcy5fYXN5bmNTdG9yYWdlTmF0aXZlTW9kdWxlLm11bHRpU2V0KFt2YWx1ZXNBcnJheV0sIGZ1bmN0aW9uKFxuICAgICAgICBlcnJvcnM6IEFycmF5PEVycm9yPixcbiAgICAgICkge1xuICAgICAgICBjb25zdCBlcnJzID0gY29udmVydEVycm9ycyhlcnJvcnMpO1xuICAgICAgICBpZiAoZXJycyAmJiBlcnJzLmxlbmd0aCkge1xuICAgICAgICAgIHJlamVjdChlcnJzWzBdKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICByZXNvbHZlKCk7XG4gICAgICAgIH1cbiAgICAgIH0pO1xuICAgIH0pO1xuICB9XG5cbiAgYXN5bmMgcmVtb3ZlU2luZ2xlKGtleToga2V5b2YgVCwgXz86IFN0b3JhZ2VPcHRpb25zKTogUHJvbWlzZTx2b2lkPiB7XG4gICAgY2hlY2tWYWxpZElucHV0KGtleSk7XG5cbiAgICByZXR1cm4gbmV3IFByb21pc2U8dm9pZD4oKHJlc29sdmUsIHJlamVjdCkgPT4ge1xuICAgICAgdGhpcy5fYXN5bmNTdG9yYWdlTmF0aXZlTW9kdWxlLm11bHRpUmVtb3ZlKFtrZXldLCBmdW5jdGlvbihcbiAgICAgICAgZXJyb3JzOiBBcnJheTxFcnJvcj4sXG4gICAgICApIHtcbiAgICAgICAgY29uc3QgZXJycyA9IGNvbnZlcnRFcnJvcnMoZXJyb3JzKTtcbiAgICAgICAgaWYgKGVycnMgJiYgZXJycy5sZW5ndGgpIHtcbiAgICAgICAgICByZWplY3QoZXJyc1swXSk7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgcmVzb2x2ZSgpO1xuICAgICAgICB9XG4gICAgICB9KTtcbiAgICB9KTtcbiAgfVxuXG4gIGFzeW5jIHJlbW92ZU1hbnkoa2V5czogQXJyYXk8a2V5b2YgVD4sIF8/OiBTdG9yYWdlT3B0aW9ucyk6IFByb21pc2U8dm9pZD4ge1xuICAgIGtleXMuZm9yRWFjaChrID0+IGNoZWNrVmFsaWRJbnB1dChrKSk7XG5cbiAgICByZXR1cm4gbmV3IFByb21pc2U8dm9pZD4oKHJlc29sdmUsIHJlamVjdCkgPT4ge1xuICAgICAgdGhpcy5fYXN5bmNTdG9yYWdlTmF0aXZlTW9kdWxlLm11bHRpUmVtb3ZlKGtleXMsIGZ1bmN0aW9uKFxuICAgICAgICBlcnJvcnM6IEFycmF5PEVycm9yPixcbiAgICAgICkge1xuICAgICAgICBjb25zdCBlcnJzID0gY29udmVydEVycm9ycyhlcnJvcnMpO1xuICAgICAgICBpZiAoZXJycyAmJiBlcnJzLmxlbmd0aCkge1xuICAgICAgICAgIHJlamVjdChlcnJzWzBdKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICByZXNvbHZlKCk7XG4gICAgICAgIH1cbiAgICAgIH0pO1xuICAgIH0pO1xuICB9XG5cbiAgYXN5bmMgZ2V0S2V5cyhfPzogU3RvcmFnZU9wdGlvbnMpOiBQcm9taXNlPEFycmF5PGtleW9mIFQ+PiB7XG4gICAgcmV0dXJuIG5ldyBQcm9taXNlKChyZXNvbHZlLCByZWplY3QpID0+IHtcbiAgICAgIHRoaXMuX2FzeW5jU3RvcmFnZU5hdGl2ZU1vZHVsZS5nZXRBbGxLZXlzKGZ1bmN0aW9uKFxuICAgICAgICBlcnJvcnM6IEFycmF5PEVycm9yPixcbiAgICAgICAga2V5czogQXJyYXk8a2V5b2YgVD4sXG4gICAgICApIHtcbiAgICAgICAgY29uc3QgZXJyID0gY29udmVydEVycm9ycyhlcnJvcnMpO1xuXG4gICAgICAgIGlmIChlcnIgJiYgZXJyLmxlbmd0aCkge1xuICAgICAgICAgIHJlamVjdChlcnJbMF0pO1xuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIHJlc29sdmUoa2V5cyk7XG4gICAgICAgIH1cbiAgICAgIH0pO1xuICAgIH0pO1xuICB9XG5cbiAgYXN5bmMgZHJvcFN0b3JhZ2UoXz86IFN0b3JhZ2VPcHRpb25zKTogUHJvbWlzZTx2b2lkPiB7XG4gICAgcmV0dXJuIG5ldyBQcm9taXNlPHZvaWQ+KChyZXNvbHZlLCByZWplY3QpID0+IHtcbiAgICAgIHRoaXMuX2FzeW5jU3RvcmFnZU5hdGl2ZU1vZHVsZS5jbGVhcihmdW5jdGlvbihlcnJvcjogQXJyYXk8RXJyb3I+KSB7XG4gICAgICAgIGNvbnN0IGVyciA9IGNvbnZlcnRFcnJvcnMoQXJyYXkuaXNBcnJheShlcnJvcikgPyBlcnJvciA6IFtlcnJvcl0pO1xuXG4gICAgICAgIGlmIChlcnIpIHtcbiAgICAgICAgICByZWplY3QoZXJyKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICByZXNvbHZlKCk7XG4gICAgICAgIH1cbiAgICAgIH0pO1xuICAgIH0pO1xuICB9XG59XG4iXX0=