"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _defaults = require("./defaults");

var _extension = require("./extension");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class AsyncStorage {
  constructor(storageBackend, asOptions) {
    _defineProperty(this, "ext", void 0);

    _defineProperty(this, "_backend", void 0);

    _defineProperty(this, "_config", void 0);

    _defineProperty(this, "log", void 0);

    _defineProperty(this, "error", void 0);

    this._backend = storageBackend;
    this._config = asOptions; // off by default

    this.log = _defaults.noop;
    this.error = _defaults.noop;
    this.ext = (0, _extension.createExtension)(this._backend);

    if (this._config.logger) {
      this.log = typeof this._config.logger === 'function' ? this._config.logger : _defaults.simpleLogger;
    }

    if (this._config.errorHandler) {
      this.error = typeof this._config.errorHandler === 'function' ? this._config.errorHandler : _defaults.simpleErrorHandler;
    }
  }

  async get(key, opts = null) {
    let value = null;

    try {
      this.log({
        action: 'read-single',
        key: key
      });
      value = await this._backend.getSingle(key, opts);
    } catch (e) {
      this.error(e);
    }

    return value;
  }

  async set(key, value, opts = null) {
    try {
      this.log({
        action: 'save-single',
        key: key,
        value
      });
      await this._backend.setSingle(key, value, opts);
    } catch (e) {
      this.error(e);
    }
  }

  async getMultiple(keys, opts = null) {
    let values = {};

    try {
      this.log({
        action: 'read-many',
        key: keys
      });
      values = await this._backend.getMany(keys, opts);
    } catch (e) {
      this.error(e);
    }

    return values;
  }

  async setMultiple(keyValues, opts = null) {
    try {
      this.log({
        action: 'save-many',
        value: keyValues
      });
      await this._backend.setMany(keyValues, opts);
    } catch (e) {
      this.error(e);
    }
  }

  async remove(key, opts = null) {
    try {
      this.log({
        action: 'delete-single',
        key: key
      });
      await this._backend.removeSingle(key, opts);
    } catch (e) {
      this.error(e);
    }
  }

  async removeMultiple(keys, opts = null) {
    try {
      this.log({
        action: 'delete-many',
        key: keys
      });
      await this._backend.removeMany(keys, opts);
    } catch (e) {
      this.error(e);
    }
  }

  async getKeys(opts = null) {
    let keys = [];

    try {
      this.log({
        action: 'keys'
      });
      keys = await this._backend.getKeys(opts);
    } catch (e) {
      this.error(e);
    }

    return keys;
  }

  async clearStorage(opts = null) {
    try {
      this.log({
        action: 'drop'
      });
      await this._backend.dropStorage(opts);
    } catch (e) {
      this.error(e);
    }
  }

}

var _default = AsyncStorage;
exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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