(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = global || self, factory(global.IntlListFormat = {}));
}(this, (function (exports) { 'use strict';

    function invariant(condition, message, Err) {
        if (Err === void 0) { Err = Error; }
        if (!condition) {
            throw new Err(message);
        }
    }

    /**
     * https://tc39.es/ecma262/#sec-toobject
     * @param arg
     */
    function toObject(arg) {
        if (arg == null) {
            throw new TypeError('undefined/null cannot be converted to object');
        }
        return Object(arg);
    }
    /**
     * https://tc39.es/ecma262/#sec-tostring
     */
    function toString(o) {
        // Only symbol is irregular...
        if (typeof o === 'symbol') {
            throw TypeError('Cannot convert a Symbol value to a string');
        }
        return String(o);
    }
    /**
     * https://tc39.es/ecma402/#sec-getoption
     * @param opts
     * @param prop
     * @param type
     * @param values
     * @param fallback
     */
    function getOption(opts, prop, type, values, fallback) {
        // const descriptor = Object.getOwnPropertyDescriptor(opts, prop);
        var value = opts[prop];
        if (value !== undefined) {
            if (type !== 'boolean' && type !== 'string') {
                throw new TypeError('invalid type');
            }
            if (type === 'boolean') {
                value = Boolean(value);
            }
            if (type === 'string') {
                value = toString(value);
            }
            if (values !== undefined && !values.filter(function (val) { return val == value; }).length) {
                throw new RangeError(value + " is not within " + values.join(', '));
            }
            return value;
        }
        return fallback;
    }
    function setInternalSlot(map, pl, field, value) {
        if (!map.get(pl)) {
            map.set(pl, Object.create(null));
        }
        var slots = map.get(pl);
        slots[field] = value;
    }
    function getInternalSlot(map, pl, field) {
        return getMultiInternalSlots(map, pl, field)[field];
    }
    function getMultiInternalSlots(map, pl) {
        var fields = [];
        for (var _i = 2; _i < arguments.length; _i++) {
            fields[_i - 2] = arguments[_i];
        }
        var slots = map.get(pl);
        if (!slots) {
            throw new TypeError(pl + " InternalSlot has not been initialized");
        }
        return fields.reduce(function (all, f) {
            all[f] = slots[f];
            return all;
        }, Object.create(null));
    }
    function isLiteralPart(patternPart) {
        return patternPart.type === 'literal';
    }
    function partitionPattern(pattern) {
        var result = [];
        var beginIndex = pattern.indexOf('{');
        var endIndex = 0;
        var nextIndex = 0;
        var length = pattern.length;
        while (beginIndex < pattern.length && beginIndex > -1) {
            endIndex = pattern.indexOf('}', beginIndex);
            invariant(endIndex > beginIndex, "Invalid pattern " + pattern);
            if (beginIndex > nextIndex) {
                result.push({
                    type: 'literal',
                    value: pattern.substring(nextIndex, beginIndex),
                });
            }
            result.push({
                type: pattern.substring(beginIndex + 1, endIndex),
                value: undefined,
            });
            nextIndex = endIndex + 1;
            beginIndex = pattern.indexOf('{', nextIndex);
        }
        if (nextIndex < length) {
            result.push({
                type: 'literal',
                value: pattern.substring(nextIndex, length),
            });
        }
        return result;
    }

    /**
     * IE11-safe version of getCanonicalLocales since it's ES2016
     * @param locales locales
     */
    function getCanonicalLocales(locales) {
        // IE11
        var getCanonicalLocales = Intl.getCanonicalLocales;
        if (typeof getCanonicalLocales === 'function') {
            return getCanonicalLocales(locales);
        }
        return Intl.NumberFormat.supportedLocalesOf(locales || '');
    }

    var __extends = (undefined && undefined.__extends) || (function () {
        var extendStatics = function (d, b) {
            extendStatics = Object.setPrototypeOf ||
                ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
                function (d, b) { for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p]; };
            return extendStatics(d, b);
        };
        return function (d, b) {
            extendStatics(d, b);
            function __() { this.constructor = d; }
            d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
        };
    })();
    var __assign = (undefined && undefined.__assign) || function () {
        __assign = Object.assign || function(t) {
            for (var s, i = 1, n = arguments.length; i < n; i++) {
                s = arguments[i];
                for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                    t[p] = s[p];
            }
            return t;
        };
        return __assign.apply(this, arguments);
    };
    function createResolveLocale(getDefaultLocale) {
        var lookupMatcher = createLookupMatcher(getDefaultLocale);
        var bestFitMatcher = createBestFitMatcher(getDefaultLocale);
        /**
         * https://tc39.es/ecma402/#sec-resolvelocale
         */
        return function resolveLocale(availableLocales, requestedLocales, options, relevantExtensionKeys, localeData) {
            var matcher = options.localeMatcher;
            var r;
            if (matcher === 'lookup') {
                r = lookupMatcher(availableLocales, requestedLocales);
            }
            else {
                r = bestFitMatcher(availableLocales, requestedLocales);
            }
            var foundLocale = r.locale;
            var result = { locale: '', dataLocale: foundLocale };
            var supportedExtension = '-u';
            for (var _i = 0, relevantExtensionKeys_1 = relevantExtensionKeys; _i < relevantExtensionKeys_1.length; _i++) {
                var key = relevantExtensionKeys_1[_i];
                var foundLocaleData = localeData[foundLocale];
                invariant(typeof foundLocaleData === 'object' && foundLocaleData !== null, "locale data " + key + " must be an object");
                var keyLocaleData = foundLocaleData[key];
                invariant(Array.isArray(keyLocaleData), "keyLocaleData for " + key + " must be an array");
                var value = keyLocaleData[0];
                invariant(typeof value === 'string' || value === null, 'value must be string or null');
                var supportedExtensionAddition = '';
                if (r.extension) {
                    var requestedValue = unicodeExtensionValue(r.extension, key);
                    if (requestedValue !== undefined) {
                        if (requestedValue !== '') {
                            if (~keyLocaleData.indexOf(requestedValue)) {
                                value = requestedValue;
                                supportedExtensionAddition = "-" + key + "-" + value;
                            }
                        }
                        else if (~requestedValue.indexOf('true')) {
                            value = 'true';
                            supportedExtensionAddition = "-" + key;
                        }
                    }
                }
                if (key in options) {
                    var optionsValue = options[key];
                    invariant(typeof optionsValue === 'string' ||
                        typeof optionsValue === 'undefined' ||
                        optionsValue === null, 'optionsValue must be String, Undefined or Null');
                    if (~keyLocaleData.indexOf(optionsValue)) {
                        if (optionsValue !== value) {
                            value = optionsValue;
                            supportedExtensionAddition = '';
                        }
                    }
                }
                result[key] = value;
                supportedExtension += supportedExtensionAddition;
            }
            if (supportedExtension.length > 2) {
                var privateIndex = foundLocale.indexOf('-x-');
                if (privateIndex === -1) {
                    foundLocale = foundLocale + supportedExtension;
                }
                else {
                    var preExtension = foundLocale.slice(0, privateIndex);
                    var postExtension = foundLocale.slice(privateIndex, foundLocale.length);
                    foundLocale = preExtension + supportedExtension + postExtension;
                }
                foundLocale = getCanonicalLocales(foundLocale)[0];
            }
            result.locale = foundLocale;
            return result;
        };
    }
    /**
     * https://tc39.es/ecma402/#sec-unicodeextensionvalue
     * @param extension
     * @param key
     */
    function unicodeExtensionValue(extension, key) {
        invariant(key.length === 2, 'key must have 2 elements');
        var size = extension.length;
        var searchValue = "-" + key + "-";
        var pos = extension.indexOf(searchValue);
        if (pos !== -1) {
            var start = pos + 4;
            var end = start;
            var k = start;
            var done = false;
            while (!done) {
                var e = extension.indexOf('-', k);
                var len = void 0;
                if (e === -1) {
                    len = size - k;
                }
                else {
                    len = e - k;
                }
                if (len === 2) {
                    done = true;
                }
                else if (e === -1) {
                    end = size;
                    done = true;
                }
                else {
                    end = e;
                    k = e + 1;
                }
            }
            return extension.slice(start, end);
        }
        searchValue = "-" + key;
        pos = extension.indexOf(searchValue);
        if (pos !== -1 && pos + 3 === size) {
            return '';
        }
        return undefined;
    }
    var UNICODE_EXTENSION_SEQUENCE_REGEX = /-u(?:-[0-9a-z]{2,8})+/gi;
    /**
     * https://tc39.es/ecma402/#sec-bestavailablelocale
     * @param availableLocales
     * @param locale
     */
    function bestAvailableLocale(availableLocales, locale) {
        var candidate = locale;
        while (true) {
            if (~availableLocales.indexOf(candidate)) {
                return candidate;
            }
            var pos = candidate.lastIndexOf('-');
            if (!~pos) {
                return undefined;
            }
            if (pos >= 2 && candidate[pos - 2] === '-') {
                pos -= 2;
            }
            candidate = candidate.slice(0, pos);
        }
    }
    function createLookupMatcher(getDefaultLocale) {
        /**
         * https://tc39.es/ecma402/#sec-lookupmatcher
         */
        return function lookupMatcher(availableLocales, requestedLocales) {
            var result = { locale: '' };
            for (var _i = 0, requestedLocales_1 = requestedLocales; _i < requestedLocales_1.length; _i++) {
                var locale = requestedLocales_1[_i];
                var noExtensionLocale = locale.replace(UNICODE_EXTENSION_SEQUENCE_REGEX, '');
                var availableLocale = bestAvailableLocale(availableLocales, noExtensionLocale);
                if (availableLocale) {
                    result.locale = availableLocale;
                    if (locale !== noExtensionLocale) {
                        result.extension = locale.slice(noExtensionLocale.length + 1, locale.length);
                    }
                    return result;
                }
            }
            result.locale = getDefaultLocale();
            return result;
        };
    }
    function createBestFitMatcher(getDefaultLocale) {
        return function bestFitMatcher(availableLocales, requestedLocales) {
            var result = { locale: '' };
            for (var _i = 0, requestedLocales_2 = requestedLocales; _i < requestedLocales_2.length; _i++) {
                var locale = requestedLocales_2[_i];
                var noExtensionLocale = locale.replace(UNICODE_EXTENSION_SEQUENCE_REGEX, '');
                var availableLocale = bestAvailableLocale(availableLocales, noExtensionLocale);
                if (availableLocale) {
                    result.locale = availableLocale;
                    if (locale !== noExtensionLocale) {
                        result.extension = locale.slice(noExtensionLocale.length + 1, locale.length);
                    }
                    return result;
                }
            }
            result.locale = getDefaultLocale();
            return result;
        };
    }
    function getLocaleHierarchy(locale, aliases, parentLocales) {
        var results = [locale];
        if (aliases[locale]) {
            locale = aliases[locale];
            results.push(locale);
        }
        var parentLocale = parentLocales[locale];
        if (parentLocale) {
            results.push(parentLocale);
        }
        var localeParts = locale.split('-');
        for (var i = localeParts.length; i > 1; i--) {
            results.push(localeParts.slice(0, i - 1).join('-'));
        }
        return results;
    }
    function lookupSupportedLocales(availableLocales, requestedLocales) {
        var subset = [];
        for (var _i = 0, requestedLocales_3 = requestedLocales; _i < requestedLocales_3.length; _i++) {
            var locale = requestedLocales_3[_i];
            var noExtensionLocale = locale.replace(UNICODE_EXTENSION_SEQUENCE_REGEX, '');
            var availableLocale = bestAvailableLocale(availableLocales, noExtensionLocale);
            if (availableLocale) {
                subset.push(availableLocale);
            }
        }
        return subset;
    }
    function supportedLocales(availableLocales, requestedLocales, options) {
        var matcher = 'best fit';
        if (options !== undefined) {
            options = toObject(options);
            matcher = getOption(options, 'localeMatcher', 'string', ['lookup', 'best fit'], 'best fit');
        }
        if (matcher === 'best fit') {
            return lookupSupportedLocales(availableLocales, requestedLocales);
        }
        return lookupSupportedLocales(availableLocales, requestedLocales);
    }
    var MissingLocaleDataError = /** @class */ (function (_super) {
        __extends(MissingLocaleDataError, _super);
        function MissingLocaleDataError() {
            var _this = _super !== null && _super.apply(this, arguments) || this;
            _this.type = 'MISSING_LOCALE_DATA';
            return _this;
        }
        return MissingLocaleDataError;
    }(Error));
    function unpackData(locale, localeData, 
    /** By default shallow merge the dictionaries. */
    reducer) {
        if (reducer === void 0) { reducer = function (all, d) { return (__assign(__assign({}, all), d)); }; }
        var localeHierarchy = getLocaleHierarchy(locale, localeData.aliases, localeData.parentLocales);
        var dataToMerge = localeHierarchy
            .map(function (l) { return localeData.data[l]; })
            .filter(Boolean);
        if (!dataToMerge.length) {
            throw new MissingLocaleDataError("Missing locale data for \"" + locale + "\", lookup hierarchy: " + localeHierarchy.join(', '));
        }
        dataToMerge.reverse();
        return dataToMerge.reduce(reducer, {});
    }

    var __assign$1 = (undefined && undefined.__assign) || function () {
        __assign$1 = Object.assign || function(t) {
            for (var s, i = 1, n = arguments.length; i < n; i++) {
                s = arguments[i];
                for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                    t[p] = s[p];
            }
            return t;
        };
        return __assign$1.apply(this, arguments);
    };
    var __spreadArrays = (undefined && undefined.__spreadArrays) || function () {
        for (var s = 0, i = 0, il = arguments.length; i < il; i++) s += arguments[i].length;
        for (var r = Array(s), k = 0, i = 0; i < il; i++)
            for (var a = arguments[i], j = 0, jl = a.length; j < jl; j++, k++)
                r[k] = a[j];
        return r;
    };
    function validateInstance(instance, method) {
        if (!(instance instanceof ListFormat)) {
            throw new TypeError("Method Intl.ListFormat.prototype." + method + " called on incompatible receiver " + String(instance));
        }
    }
    /**
     * https://tc39.es/proposal-intl-list-format/#sec-createstringlistfromiterable
     * @param list list
     */
    function stringListFromIterable(list) {
        if (list === undefined) {
            return [];
        }
        var result = [];
        for (var _i = 0, list_1 = list; _i < list_1.length; _i++) {
            var el = list_1[_i];
            if (typeof el !== 'string') {
                throw new TypeError("array list[" + list.indexOf(el) + "] is not type String");
            }
            result.push(el);
        }
        return result;
    }
    function createPartsFromList(internalSlotMap, lf, list) {
        var size = list.length;
        if (size === 0) {
            return [];
        }
        if (size === 2) {
            var pattern = getInternalSlot(internalSlotMap, lf, 'templatePair');
            var first = { type: 'element', value: list[0] };
            var second = { type: 'element', value: list[1] };
            return deconstructPattern(pattern, { '0': first, '1': second });
        }
        var last = {
            type: 'element',
            value: list[size - 1],
        };
        var parts = last;
        var i = size - 2;
        while (i >= 0) {
            var pattern = void 0;
            if (i === 0) {
                pattern = getInternalSlot(internalSlotMap, lf, 'templateStart');
            }
            else if (i < size - 2) {
                pattern = getInternalSlot(internalSlotMap, lf, 'templateMiddle');
            }
            else {
                pattern = getInternalSlot(internalSlotMap, lf, 'templateEnd');
            }
            var head = { type: 'element', value: list[i] };
            parts = deconstructPattern(pattern, { '0': head, '1': parts });
            i--;
        }
        return parts;
    }
    function deconstructPattern(pattern, placeables) {
        var patternParts = partitionPattern(pattern);
        var result = [];
        for (var _i = 0, patternParts_1 = patternParts; _i < patternParts_1.length; _i++) {
            var patternPart = patternParts_1[_i];
            var part = patternPart.type;
            if (isLiteralPart(patternPart)) {
                result.push({
                    type: 'literal',
                    value: patternPart.value,
                });
            }
            else {
                invariant(part in placeables, part + " is missing from placables");
                var subst = placeables[part];
                if (Array.isArray(subst)) {
                    result.push.apply(result, subst);
                }
                else {
                    result.push(subst);
                }
            }
        }
        return result;
    }
    var ListFormat = /** @class */ (function () {
        function ListFormat(locales, options) {
            // test262/test/intl402/ListFormat/constructor/constructor/newtarget-undefined.js
            // Cannot use `new.target` bc of IE11 & TS transpiles it to something else
            var newTarget = this && this instanceof ListFormat ? this.constructor : void 0;
            if (!newTarget) {
                throw new TypeError("Intl.ListFormat must be called with 'new'");
            }
            setInternalSlot(ListFormat.__INTERNAL_SLOT_MAP__, this, 'initializedListFormat', true);
            var requestedLocales = getCanonicalLocales(locales);
            var opt = Object.create(null);
            var opts = options === undefined ? Object.create(null) : toObject(options);
            var matcher = getOption(opts, 'localeMatcher', 'string', ['best fit', 'lookup'], 'best fit');
            opt.localeMatcher = matcher;
            var localeData = ListFormat.localeData;
            var r = createResolveLocale(ListFormat.getDefaultLocale)(ListFormat.availableLocales, requestedLocales, opt, ListFormat.relevantExtensionKeys, localeData);
            setInternalSlot(ListFormat.__INTERNAL_SLOT_MAP__, this, 'locale', r.locale);
            var type = getOption(opts, 'type', 'string', ['conjunction', 'disjunction', 'unit'], 'conjunction');
            setInternalSlot(ListFormat.__INTERNAL_SLOT_MAP__, this, 'type', type);
            var style = getOption(opts, 'style', 'string', ['long', 'short', 'narrow'], 'long');
            setInternalSlot(ListFormat.__INTERNAL_SLOT_MAP__, this, 'style', style);
            var dataLocale = r.dataLocale;
            var dataLocaleData = localeData[dataLocale];
            var dataLocaleTypes = dataLocaleData[type];
            var templates = dataLocaleTypes[style];
            setInternalSlot(ListFormat.__INTERNAL_SLOT_MAP__, this, 'templatePair', templates.pair);
            setInternalSlot(ListFormat.__INTERNAL_SLOT_MAP__, this, 'templateStart', templates.start);
            setInternalSlot(ListFormat.__INTERNAL_SLOT_MAP__, this, 'templateMiddle', templates.middle);
            setInternalSlot(ListFormat.__INTERNAL_SLOT_MAP__, this, 'templateEnd', templates.end);
        }
        ListFormat.prototype.format = function (elements) {
            validateInstance(this, 'format');
            var result = '';
            var parts = createPartsFromList(ListFormat.__INTERNAL_SLOT_MAP__, this, stringListFromIterable(elements));
            if (!Array.isArray(parts)) {
                return parts.value;
            }
            for (var _i = 0, parts_1 = parts; _i < parts_1.length; _i++) {
                var p = parts_1[_i];
                result += p.value;
            }
            return result;
        };
        ListFormat.prototype.formatToParts = function (elements) {
            validateInstance(this, 'format');
            var parts = createPartsFromList(ListFormat.__INTERNAL_SLOT_MAP__, this, stringListFromIterable(elements));
            if (!Array.isArray(parts)) {
                return [parts];
            }
            var result = [];
            for (var _i = 0, parts_2 = parts; _i < parts_2.length; _i++) {
                var part = parts_2[_i];
                result.push(__assign$1({}, part));
            }
            return result;
        };
        ListFormat.prototype.resolvedOptions = function () {
            validateInstance(this, 'resolvedOptions');
            return {
                locale: getInternalSlot(ListFormat.__INTERNAL_SLOT_MAP__, this, 'locale'),
                type: getInternalSlot(ListFormat.__INTERNAL_SLOT_MAP__, this, 'type'),
                style: getInternalSlot(ListFormat.__INTERNAL_SLOT_MAP__, this, 'style'),
            };
        };
        ListFormat.supportedLocalesOf = function (locales, options) {
            // test262/test/intl402/ListFormat/constructor/supportedLocalesOf/result-type.js
            return supportedLocales(ListFormat.availableLocales, getCanonicalLocales(locales), options);
        };
        ListFormat.__addLocaleData = function () {
            var data = [];
            for (var _i = 0; _i < arguments.length; _i++) {
                data[_i] = arguments[_i];
            }
            var _loop_1 = function (datum) {
                var availableLocales = Object.keys(__spreadArrays(datum.availableLocales, Object.keys(datum.aliases), Object.keys(datum.parentLocales)).reduce(function (all, k) {
                    all[k] = true;
                    return all;
                }, {}));
                availableLocales.forEach(function (locale) {
                    try {
                        ListFormat.localeData[locale] = unpackData(locale, datum);
                    }
                    catch (e) {
                        // If we can't unpack this data, ignore the locale
                    }
                });
            };
            for (var _a = 0, data_1 = data; _a < data_1.length; _a++) {
                var datum = data_1[_a];
                _loop_1(datum);
            }
            ListFormat.availableLocales = Object.keys(ListFormat.localeData);
            if (!ListFormat.__defaultLocale) {
                ListFormat.__defaultLocale = ListFormat.availableLocales[0];
            }
        };
        ListFormat.getDefaultLocale = function () {
            return ListFormat.__defaultLocale;
        };
        ListFormat.localeData = {};
        ListFormat.availableLocales = [];
        ListFormat.__defaultLocale = 'en';
        ListFormat.relevantExtensionKeys = [];
        ListFormat.polyfilled = true;
        ListFormat.__INTERNAL_SLOT_MAP__ = new WeakMap();
        return ListFormat;
    }());
    try {
        // IE11 does not have Symbol
        if (typeof Symbol !== 'undefined') {
            Object.defineProperty(ListFormat.prototype, Symbol.toStringTag, {
                value: 'Intl.ListFormat',
                writable: false,
                enumerable: false,
                configurable: true,
            });
        }
        // https://github.com/tc39/test262/blob/master/test/intl402/ListFormat/constructor/length.js
        Object.defineProperty(ListFormat.prototype.constructor, 'length', {
            value: 0,
            writable: false,
            enumerable: false,
            configurable: true,
        });
        // https://github.com/tc39/test262/blob/master/test/intl402/ListFormat/constructor/supportedLocalesOf/length.js
        Object.defineProperty(ListFormat.supportedLocalesOf, 'length', {
            value: 1,
            writable: false,
            enumerable: false,
            configurable: true,
        });
    }
    catch (e) {
        // Meta fix so we're test262-compliant, not important
    }

    /* @generated */
    ListFormat.__addLocaleData({ "data": { "af": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} en {1}", "pair": "{0} en {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} en {1}", "pair": "{0} en {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} en {1}", "pair": "{0} en {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} of {1}", "pair": "{0} of {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} of {1}", "pair": "{0} of {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} of {1}", "pair": "{0} of {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} en {1}", "pair": "{0} en {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} en {1}", "pair": "{0} en {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} en {1}", "pair": "{0} en {1}" } } } }, "availableLocales": ["af-NA", "af"], "aliases": {}, "parentLocales": {} }, { "data": { "agq": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["agq"], "aliases": {}, "parentLocales": {} }, { "data": { "ak": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["ak"], "aliases": {}, "parentLocales": {} }, { "data": { "am": { "conjunction": { "long": { "start": "{0}፣ {1}", "middle": "{0}፣ {1}", "end": "{0}, እና {1}", "pair": "{0} እና {1}" }, "short": { "start": "{0}፣ {1}", "middle": "{0}፣ {1}", "end": "{0}, እና {1}", "pair": "{0} እና {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}፣ {1}", "end": "{0}, እና {1}", "pair": "{0} እና {1}" } }, "disjunction": { "long": { "start": "{0}፣ {1}", "middle": "{0}፣ {1}", "end": "{0} ወይም {1}﻿", "pair": "{0} ወይም {1}﻿" }, "short": { "start": "{0}፣ {1}", "middle": "{0}፣ {1}", "end": "{0} ወይም {1}﻿", "pair": "{0} ወይም {1}﻿" }, "narrow": { "start": "{0}፣ {1}", "middle": "{0}፣ {1}", "end": "{0} ወይም {1}﻿", "pair": "{0} ወይም {1}﻿" } }, "unit": { "long": { "start": "{0}፣ {1}", "middle": "{0}፣ {1}", "end": "{0}፣ {1}", "pair": "{0}፣ {1}" }, "short": { "start": "{0}፣ {1}", "middle": "{0}፣ {1}", "end": "{0}፣ {1}", "pair": "{0}፣ {1}" }, "narrow": { "start": "{0}፣ {1}", "middle": "{0}፣ {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["am"], "aliases": {}, "parentLocales": {} }, { "data": { "ar": { "conjunction": { "long": { "start": "{0} و{1}", "middle": "{0} و{1}", "end": "{0} و{1}", "pair": "{0} و{1}" }, "short": { "start": "{0} و{1}", "middle": "{0} و{1}", "end": "{0} و{1}", "pair": "{0} و{1}" }, "narrow": { "start": "{0} و{1}", "middle": "{0} و{1}", "end": "{0} و{1}", "pair": "{0} و{1}" } }, "disjunction": { "long": { "start": "{0} أو {1}", "middle": "{0} أو {1}", "end": "{0} أو {1}", "pair": "{0} أو {1}" }, "short": { "start": "{0} أو {1}", "middle": "{0} أو {1}", "end": "{0} أو {1}", "pair": "{0} أو {1}" }, "narrow": { "start": "{0} أو {1}", "middle": "{0} أو {1}", "end": "{0} أو {1}", "pair": "{0} أو {1}" } }, "unit": { "long": { "start": "{0}، و{1}", "middle": "{0}، و{1}", "end": "{0}، و{1}", "pair": "{0} و{1}" }, "short": { "start": "{0}، و{1}", "middle": "{0}، و{1}", "end": "{0}، و{1}", "pair": "{0} و{1}" }, "narrow": { "start": "{0} و{1}", "middle": "{0} و{1}", "end": "{0} و{1}", "pair": "{0} و{1}" } } } }, "availableLocales": ["ar-AE", "ar-BH", "ar-DJ", "ar-DZ", "ar-EG", "ar-EH", "ar-ER", "ar-IL", "ar-IQ", "ar-JO", "ar-KM", "ar-KW", "ar-LB", "ar-LY", "ar-MA", "ar-MR", "ar-OM", "ar-PS", "ar-QA", "ar-SA", "ar-SD", "ar-SO", "ar-SS", "ar-SY", "ar-TD", "ar-TN", "ar-YE", "ar"], "aliases": {}, "parentLocales": {} }, { "data": { "as": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} আৰু {1}", "pair": "{0} আৰু {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} আৰু {1}", "pair": "{0} আৰু {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} বা {1}", "pair": "{0} বা {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} বা {1}", "pair": "{0} বা {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} বা {1}", "pair": "{0} বা {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["as"], "aliases": {}, "parentLocales": {} }, { "data": { "asa": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["asa"], "aliases": {}, "parentLocales": {} }, { "data": { "ast": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} y {1}", "pair": "{0} y {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} y {1}", "pair": "{0} y {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} y {1}", "pair": "{0} y {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} o {1}", "pair": "{0} o {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} o {1}", "pair": "{0} o {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} o {1}", "pair": "{0} o {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} y {1}", "pair": "{0} y {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} y {1}", "pair": "{0} y {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} y {1}", "pair": "{0} y {1}" } } } }, "availableLocales": ["ast"], "aliases": {}, "parentLocales": {} }, { "data": { "az-Cyrl": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } } }, "az": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} və {1}", "pair": "{0} və {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} və {1}", "pair": "{0} və {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, yaxud {1}", "pair": "{0} yaxud {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, yaxud {1}", "pair": "{0}, yaxud {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, yaxud {1}", "pair": "{0}, yaxud {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["az-Cyrl", "az-Latn", "az"], "aliases": { "az-AZ": "az-Latn-AZ" }, "parentLocales": {} }, { "data": { "bas": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["bas"], "aliases": {}, "parentLocales": {} }, { "data": { "be": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} і {1}", "pair": "{0} і {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} і {1}", "pair": "{0} і {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} і {1}", "pair": "{0} і {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ці {1}", "pair": "{0} ці {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ці {1}", "pair": "{0} ці {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ці {1}", "pair": "{0} ці {1}" } }, "unit": { "long": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "short": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["be"], "aliases": {}, "parentLocales": {} }, { "data": { "bem": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["bem"], "aliases": {}, "parentLocales": {} }, { "data": { "bez": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["bez"], "aliases": {}, "parentLocales": {} }, { "data": { "bg": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} или {1}", "pair": "{0} или {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} или {1}", "pair": "{0} или {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} или {1}", "pair": "{0} или {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0} и {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0} и {1}" } } } }, "availableLocales": ["bg"], "aliases": {}, "parentLocales": {} }, { "data": { "bm": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["bm"], "aliases": {}, "parentLocales": {} }, { "data": { "bn": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} এবং {1}", "pair": "{0} এবং {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} এবং {1}", "pair": "{0} এবং {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, বা {1}", "pair": "{0} বা {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, বা {1}", "pair": "{0} বা {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, বা {1}", "pair": "{0} বা {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["bn-IN", "bn"], "aliases": {}, "parentLocales": {} }, { "data": { "bo": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["bo-IN", "bo"], "aliases": {}, "parentLocales": {} }, { "data": { "br": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ha {1}", "pair": "{0} ha {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} & {1}", "pair": "{0} & {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} & {1}", "pair": "{0} & {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} pe {1}", "pair": "{0} pe {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} pe {1}", "pair": "{0} pe {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} pe {1}", "pair": "{0} pe {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["br"], "aliases": {}, "parentLocales": {} }, { "data": { "brx": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["brx"], "aliases": {}, "parentLocales": {} }, { "data": { "bs-Cyrl": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" } } }, "bs": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ili {1}", "pair": "{0} ili {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ili {1}", "pair": "{0} ili {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ili {1}", "pair": "{0} ili {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" } } } }, "availableLocales": ["bs-Cyrl", "bs-Latn", "bs"], "aliases": { "bs-BA": "bs-Latn-BA" }, "parentLocales": {} }, { "data": { "ca": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} o {1}", "pair": "{0} o {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} o {1}", "pair": "{0} o {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} o {1}", "pair": "{0} o {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" } } } }, "availableLocales": ["ca-AD", "ca-ES-VALENCIA", "ca-FR", "ca-IT", "ca"], "aliases": {}, "parentLocales": {} }, { "data": { "ccp": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} 𑄃𑄳𑄃 {1}", "pair": "{0} 𑄃𑄳𑄃 {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} 𑄃𑄳𑄃 {1}", "pair": "{0} 𑄃𑄳𑄃 {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} 𑄃𑄳𑄃 {1}", "pair": "{0} 𑄃𑄳𑄃 {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["ccp-IN", "ccp"], "aliases": {}, "parentLocales": {} }, { "data": { "ce": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["ce"], "aliases": {}, "parentLocales": {} }, { "data": { "ceb": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, at {1}", "pair": "{0} at {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, ug {1}", "pair": "{0} ug {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, o {1}", "pair": "{0} o {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, o {1}", "pair": "{0} o {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, o {1}", "pair": "{0} o {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["ceb"], "aliases": {}, "parentLocales": {} }, { "data": { "cgg": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["cgg"], "aliases": {}, "parentLocales": {} }, { "data": { "chr": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, ᎠᎴ {1}", "pair": "{0} ᎠᎴ {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, & {1}", "pair": "{0} & {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, & {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, ᎠᎴᏱᎩ {1}", "pair": "{0} ᎠᎴᏱᎩ {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, ᎠᎴᏱᎩ {1}", "pair": "{0} ᎠᎴᏱᎩ {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, ᎠᎴᏱᎩ {1}", "pair": "{0} ᎠᎴᏱᎩ {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["chr"], "aliases": {}, "parentLocales": {} }, { "data": { "ckb": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["ckb-IR", "ckb"], "aliases": {}, "parentLocales": {} }, { "data": { "cs": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} a {1}", "pair": "{0} a {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} a {1}", "pair": "{0} a {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} nebo {1}", "pair": "{0} nebo {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} nebo {1}", "pair": "{0} nebo {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} nebo {1}", "pair": "{0} nebo {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} a {1}", "pair": "{0} a {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} a {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["cs"], "aliases": {}, "parentLocales": {} }, { "data": { "cu": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["cu"], "aliases": {}, "parentLocales": {} }, { "data": { "cy": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, a(c) {1}", "pair": "{0} a(c) {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, a(c) {1}", "pair": "{0} a(c) {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} neu {1}", "pair": "{0} neu {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} neu {1}", "pair": "{0} neu {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} neu {1}", "pair": "{0} neu {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["cy"], "aliases": {}, "parentLocales": {} }, { "data": { "da": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} eller {1}", "pair": "{0} eller {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} el. {1}", "pair": "{0} el. {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} el. {1}", "pair": "{0} el. {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" } } } }, "availableLocales": ["da-GL", "da"], "aliases": {}, "parentLocales": {} }, { "data": { "dav": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["dav"], "aliases": {}, "parentLocales": {} }, { "data": { "de": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} und {1}", "pair": "{0} und {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} und {1}", "pair": "{0} und {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} und {1}", "pair": "{0} und {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} oder {1}", "pair": "{0} oder {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} oder {1}", "pair": "{0} oder {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} oder {1}", "pair": "{0} oder {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} und {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} und {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} und {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["de-AT", "de-BE", "de-CH", "de-IT", "de-LI", "de-LU", "de"], "aliases": {}, "parentLocales": {} }, { "data": { "dje": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["dje"], "aliases": {}, "parentLocales": {} }, { "data": { "dsb": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} a {1}", "pair": "{0} a {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} a {1}", "pair": "{0} a {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} a {1}", "pair": "{0} a {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} a {1}", "pair": "{0} a {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} a {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["dsb"], "aliases": {}, "parentLocales": {} }, { "data": { "dua": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["dua"], "aliases": {}, "parentLocales": {} }, { "data": { "dyo": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["dyo"], "aliases": {}, "parentLocales": {} }, { "data": { "dz": { "conjunction": { "long": { "start": "{0} དང་ {1}", "middle": "{0} དང་ {1}", "end": "{0} དང་ {1}", "pair": "{0} དང་ {1}" }, "short": { "start": "{0} དང་ {1}", "middle": "{0} དང་ {1}", "end": "{0} དང་ {1}", "pair": "{0} དང་ {1}" }, "narrow": { "start": "{0} དང་ {1}", "middle": "{0} དང་ {1}", "end": "{0} དང་ {1}", "pair": "{0} དང་ {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0} དང་ {1}", "middle": "{0} དང་ {1}", "end": "{0} དང་ {1}", "pair": "{0} དང་ {1}" }, "short": { "start": "{0} དང་ {1}", "middle": "{0} དང་ {1}", "end": "{0} དང་ {1}", "pair": "{0} དང་ {1}" }, "narrow": { "start": "{0} དང་ {1}", "middle": "{0} དང་ {1}", "end": "{0} དང་ {1}", "pair": "{0} དང་ {1}" } } } }, "availableLocales": ["dz"], "aliases": {}, "parentLocales": {} }, { "data": { "ebu": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["ebu"], "aliases": {}, "parentLocales": {} }, { "data": { "ee": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, kple {1}", "pair": "{0} kple {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, kple {1}", "pair": "{0} kple {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, kple {1}", "pair": "{0} kple {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, kple {1}", "pair": "{0} kple {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, kple {1}", "pair": "{0} kple {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, kple {1}", "pair": "{0} kple {1}" } } } }, "availableLocales": ["ee-TG", "ee"], "aliases": {}, "parentLocales": {} }, { "data": { "el": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} και {1}", "pair": "{0} και {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} και {1}", "pair": "{0} και {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ή {1}", "pair": "{0} ή {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ή {1}", "pair": "{0} ή {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ή {1}", "pair": "{0} ή {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["el-CY", "el"], "aliases": {}, "parentLocales": {} }, { "data": { "en-001": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} and {1}", "pair": "{0} and {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} and {1}", "pair": "{0} and {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} or {1}", "pair": "{0} or {1}" } } }, "en-150": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} and {1}", "pair": "{0} and {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} and {1}", "pair": "{0} and {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} or {1}", "pair": "{0} or {1}" } } }, "en-CA": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} and {1}", "pair": "{0} and {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, and {1}", "pair": "{0} and {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, and {1}", "pair": "{0} and {1}" } } }, "en-IN": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} and {1}", "pair": "{0} and {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} and {1}", "pair": "{0} and {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, and {1}", "pair": "{0}, {1}" } } }, "en": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, and {1}", "pair": "{0} and {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, & {1}", "pair": "{0} & {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["en-001", "en-150", "en-AE", "en-AG", "en-AI", "en-AS", "en-AT", "en-AU", "en-BB", "en-BE", "en-BI", "en-BM", "en-BS", "en-BW", "en-BZ", "en-CA", "en-CC", "en-CH", "en-CK", "en-CM", "en-CX", "en-CY", "en-DE", "en-DG", "en-DK", "en-DM", "en-ER", "en-FI", "en-FJ", "en-FK", "en-FM", "en-GB", "en-GD", "en-GG", "en-GH", "en-GI", "en-GM", "en-GU", "en-GY", "en-HK", "en-IE", "en-IL", "en-IM", "en-IN", "en-IO", "en-JE", "en-JM", "en-KE", "en-KI", "en-KN", "en-KY", "en-LC", "en-LR", "en-LS", "en-MG", "en-MH", "en-MO", "en-MP", "en-MS", "en-MT", "en-MU", "en-MW", "en-MY", "en-NA", "en-NF", "en-NG", "en-NL", "en-NR", "en-NU", "en-NZ", "en-PG", "en-PH", "en-PK", "en-PN", "en-PR", "en-PW", "en-RW", "en-SB", "en-SC", "en-SD", "en-SE", "en-SG", "en-SH", "en-SI", "en-SL", "en-SS", "en-SX", "en-SZ", "en-TC", "en-TK", "en-TO", "en-TT", "en-TV", "en-TZ", "en-UG", "en-UM", "en-US-POSIX", "en-US", "en-VC", "en-VG", "en-VI", "en-VU", "en-WS", "en-ZA", "en-ZM", "en-ZW", "en"], "aliases": {}, "parentLocales": { "en-150": "en-001", "en-AG": "en-001", "en-AI": "en-001", "en-AU": "en-001", "en-BB": "en-001", "en-BM": "en-001", "en-BS": "en-001", "en-BW": "en-001", "en-BZ": "en-001", "en-CA": "en-001", "en-CC": "en-001", "en-CK": "en-001", "en-CM": "en-001", "en-CX": "en-001", "en-CY": "en-001", "en-DG": "en-001", "en-DM": "en-001", "en-ER": "en-001", "en-FJ": "en-001", "en-FK": "en-001", "en-FM": "en-001", "en-GB": "en-001", "en-GD": "en-001", "en-GG": "en-001", "en-GH": "en-001", "en-GI": "en-001", "en-GM": "en-001", "en-GY": "en-001", "en-HK": "en-001", "en-IE": "en-001", "en-IL": "en-001", "en-IM": "en-001", "en-IN": "en-001", "en-IO": "en-001", "en-JE": "en-001", "en-JM": "en-001", "en-KE": "en-001", "en-KI": "en-001", "en-KN": "en-001", "en-KY": "en-001", "en-LC": "en-001", "en-LR": "en-001", "en-LS": "en-001", "en-MG": "en-001", "en-MO": "en-001", "en-MS": "en-001", "en-MT": "en-001", "en-MU": "en-001", "en-MW": "en-001", "en-MY": "en-001", "en-NA": "en-001", "en-NF": "en-001", "en-NG": "en-001", "en-NR": "en-001", "en-NU": "en-001", "en-NZ": "en-001", "en-PG": "en-001", "en-PH": "en-001", "en-PK": "en-001", "en-PN": "en-001", "en-PW": "en-001", "en-RW": "en-001", "en-SB": "en-001", "en-SC": "en-001", "en-SD": "en-001", "en-SG": "en-001", "en-SH": "en-001", "en-SL": "en-001", "en-SS": "en-001", "en-SX": "en-001", "en-SZ": "en-001", "en-TC": "en-001", "en-TK": "en-001", "en-TO": "en-001", "en-TT": "en-001", "en-TV": "en-001", "en-TZ": "en-001", "en-UG": "en-001", "en-VC": "en-001", "en-VG": "en-001", "en-VU": "en-001", "en-WS": "en-001", "en-ZA": "en-001", "en-ZM": "en-001", "en-ZW": "en-001", "en-AT": "en-150", "en-BE": "en-150", "en-CH": "en-150", "en-DE": "en-150", "en-DK": "en-150", "en-FI": "en-150", "en-NL": "en-150", "en-SE": "en-150", "en-SI": "en-150" } }, { "data": { "eo": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["eo"], "aliases": {}, "parentLocales": {} }, { "data": { "es-DO": { "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} y {1}", "pair": "{0} y {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} y {1}", "pair": "{0} y {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} y {1}", "pair": "{0} {1}" } } }, "es-PY": { "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} y {1}", "pair": "{0} y {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} y {1}", "pair": "{0} y {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } }, "es": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} y {1}", "pair": "{0} y {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} y {1}", "pair": "{0} y {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} y {1}", "pair": "{0} y {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} o {1}", "pair": "{0} o {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} o {1}", "pair": "{0} o {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} o {1}", "pair": "{0} o {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} y {1}", "pair": "{0} y {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0} y {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["es-419", "es-AR", "es-BO", "es-BR", "es-BZ", "es-CL", "es-CO", "es-CR", "es-CU", "es-DO", "es-EA", "es-EC", "es-GQ", "es-GT", "es-HN", "es-IC", "es-MX", "es-NI", "es-PA", "es-PE", "es-PH", "es-PR", "es-PY", "es-SV", "es-US", "es-UY", "es-VE", "es"], "aliases": {}, "parentLocales": { "es-AR": "es-419", "es-BO": "es-419", "es-BR": "es-419", "es-BZ": "es-419", "es-CL": "es-419", "es-CO": "es-419", "es-CR": "es-419", "es-CU": "es-419", "es-DO": "es-419", "es-EC": "es-419", "es-GT": "es-419", "es-HN": "es-419", "es-MX": "es-419", "es-NI": "es-419", "es-PA": "es-419", "es-PE": "es-419", "es-PR": "es-419", "es-PY": "es-419", "es-SV": "es-419", "es-US": "es-419", "es-UY": "es-419", "es-VE": "es-419" } }, { "data": { "et": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ja {1}", "pair": "{0} ja {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ja {1}", "pair": "{0} ja {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} või {1}", "pair": "{0} või {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} või {1}", "pair": "{0} või {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} või {1}", "pair": "{0} või {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["et"], "aliases": {}, "parentLocales": {} }, { "data": { "eu": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} eta {1}", "pair": "{0} eta {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} eta {1}", "pair": "{0} eta {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} edo {1}", "pair": "{0} edo {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} edo {1}", "pair": "{0} edo {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} edo {1}", "pair": "{0} edo {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} eta {1}", "pair": "{0} eta {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} eta {1}", "pair": "{0} eta {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} eta {1}", "pair": "{0} eta {1}" } } } }, "availableLocales": ["eu"], "aliases": {}, "parentLocales": {} }, { "data": { "ewo": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["ewo"], "aliases": {}, "parentLocales": {} }, { "data": { "fa": { "conjunction": { "long": { "start": "{0}،‏ {1}", "middle": "{0}،‏ {1}", "end": "{0}، و {1}", "pair": "{0} و {1}" }, "short": { "start": "{0}،‏ {1}", "middle": "{0}،‏ {1}", "end": "{0}، و {1}", "pair": "{0} و {1}" }, "narrow": { "start": "{0}،‏ {1}", "middle": "{0}،‏ {1}", "end": "{0}،‏ {1}", "pair": "{0}،‏ {1}" } }, "disjunction": { "long": { "start": "{0}،‏ {1}", "middle": "{0}،‏ {1}", "end": "{0}، یا {1}", "pair": "{0} یا {1}" }, "short": { "start": "{0}،‏ {1}", "middle": "{0}،‏ {1}", "end": "{0}، یا {1}", "pair": "{0} یا {1}" }, "narrow": { "start": "{0}،‏ {1}", "middle": "{0}،‏ {1}", "end": "{0}، یا {1}", "pair": "{0} یا {1}" } }, "unit": { "long": { "start": "{0}،‏ {1}", "middle": "{0}،‏ {1}", "end": "{0}، و {1}", "pair": "{0} و {1}" }, "short": { "start": "{0}،‏ {1}", "middle": "{0}،‏ {1}", "end": "{0}،‏ {1}", "pair": "{0}،‏ {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["fa-AF", "fa"], "aliases": {}, "parentLocales": {} }, { "data": { "ff": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["ff-Latn-BF", "ff-Latn-CM", "ff-Latn-GH", "ff-Latn-GM", "ff-Latn-GN", "ff-Latn-GW", "ff-Latn-LR", "ff-Latn-MR", "ff-Latn-NE", "ff-Latn-NG", "ff-Latn-SL", "ff-Latn", "ff"], "aliases": {}, "parentLocales": {} }, { "data": { "fi": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ja {1}", "pair": "{0} ja {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ja {1}", "pair": "{0} ja {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ja {1}", "pair": "{0} ja {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} tai {1}", "pair": "{0} tai {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} tai {1}", "pair": "{0} tai {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} tai {1}", "pair": "{0} tai {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ja {1}", "pair": "{0} ja {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["fi"], "aliases": {}, "parentLocales": {} }, { "data": { "fil": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, at {1}", "pair": "{0} at {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, at {1}", "pair": "{0} at {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, o {1}", "pair": "{0} o {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, o {1}", "pair": "{0} o {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, o {1}", "pair": "{0} o {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["fil"], "aliases": {}, "parentLocales": {} }, { "data": { "fo": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, ella {1}", "pair": "{0} ella {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, ella {1}", "pair": "{0} ella {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, ella {1}", "pair": "{0} ella {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["fo-DK", "fo"], "aliases": {}, "parentLocales": {} }, { "data": { "fr": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} et {1}", "pair": "{0} et {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} et {1}", "pair": "{0} et {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ou {1}", "pair": "{0} ou {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ou {1}", "pair": "{0} ou {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ou {1}", "pair": "{0} ou {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} et {1}", "pair": "{0} et {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} et {1}", "pair": "{0} et {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["fr-BE", "fr-BF", "fr-BI", "fr-BJ", "fr-BL", "fr-CA", "fr-CD", "fr-CF", "fr-CG", "fr-CH", "fr-CI", "fr-CM", "fr-DJ", "fr-DZ", "fr-GA", "fr-GF", "fr-GN", "fr-GP", "fr-GQ", "fr-HT", "fr-KM", "fr-LU", "fr-MA", "fr-MC", "fr-MF", "fr-MG", "fr-ML", "fr-MQ", "fr-MR", "fr-MU", "fr-NC", "fr-NE", "fr-PF", "fr-PM", "fr-RE", "fr-RW", "fr-SC", "fr-SN", "fr-SY", "fr-TD", "fr-TG", "fr-TN", "fr-VU", "fr-WF", "fr-YT", "fr"], "aliases": {}, "parentLocales": {} }, { "data": { "fur": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" } } } }, "availableLocales": ["fur"], "aliases": {}, "parentLocales": {} }, { "data": { "fy": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} en {1}", "pair": "{0} en {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} en {1}", "pair": "{0} en {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} en {1}", "pair": "{0} en {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0} en {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["fy"], "aliases": {}, "parentLocales": {} }, { "data": { "ga": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, agus {1}", "pair": "{0} agus {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, agus {1}", "pair": "{0} agus {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, nó {1}", "pair": "{0} nó {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, nó {1}", "pair": "{0} nó {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, nó {1}", "pair": "{0} nó {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, agus {1}", "pair": "{0} agus {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["ga-GB", "ga"], "aliases": {}, "parentLocales": {} }, { "data": { "gd": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} agus {1}", "pair": "{0} agus {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ⁊ {1}", "pair": "{0} ⁊ {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} no {1}", "pair": "{0} no {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} no {1}", "pair": "{0} no {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} no {1}", "pair": "{0} no {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} agus {1}", "pair": "{0} agus {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ’s {1}", "pair": "{0} ’s {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["gd"], "aliases": {}, "parentLocales": {} }, { "data": { "gl": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ou {1}", "pair": "{0} ou {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ou {1}", "pair": "{0} ou {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ou {1}", "pair": "{0} ou {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["gl"], "aliases": {}, "parentLocales": {} }, { "data": { "gsw": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} und {1}", "pair": "{0} und {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} und {1}", "pair": "{0} und {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} und {1}", "pair": "{0} und {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} und {1}", "pair": "{0} und {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} und {1}", "pair": "{0} und {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} und {1}", "pair": "{0} und {1}" } } } }, "availableLocales": ["gsw-FR", "gsw-LI", "gsw"], "aliases": {}, "parentLocales": {} }, { "data": { "gu": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} અને {1}", "pair": "{0} અને {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} અને {1}", "pair": "{0} અને {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, અથવા {1}", "pair": "{0} અથવા {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} અથવા {1}", "pair": "{0} અથવા {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} અથવા {1}", "pair": "{0} અથવા {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0} અને {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["gu"], "aliases": {}, "parentLocales": {} }, { "data": { "guz": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["guz"], "aliases": {}, "parentLocales": {} }, { "data": { "gv": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["gv"], "aliases": {}, "parentLocales": {} }, { "data": { "ha": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} da {1}", "pair": "{0} da {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} da {1}", "pair": "{0} da {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} da {1}", "pair": "{0} da {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ko {1}", "pair": "{0} ko {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ko {1}", "pair": "{0} ko {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ko {1}", "pair": "{0} ko {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["ha-GH", "ha-NE", "ha"], "aliases": { "ha-Latn-GH": "ha-GH", "ha-Latn-NE": "ha-NE", "ha-Latn-NG": "ha-NG" }, "parentLocales": {} }, { "data": { "haw": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["haw"], "aliases": {}, "parentLocales": {} }, { "data": { "he": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ו{1}", "pair": "{0} ו{1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ו{1}", "pair": "{0} ו{1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ו{1}", "pair": "{0} ו{1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, או {1}", "pair": "{0} או {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} או {1}", "pair": "{0} או {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} או {1}", "pair": "{0} או {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ו{1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["he"], "aliases": {}, "parentLocales": {} }, { "data": { "hi": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, और {1}", "pair": "{0} और {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} और {1}", "pair": "{0} और {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} और {1}", "pair": "{0} और {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} या {1}", "pair": "{0} या {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} या {1}", "pair": "{0} या {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} या {1}", "pair": "{0} या {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, और {1}", "pair": "{0} और {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["hi"], "aliases": {}, "parentLocales": {} }, { "data": { "hr": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ili {1}", "pair": "{0} ili {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ili {1}", "pair": "{0} ili {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ili {1}", "pair": "{0} ili {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["hr-BA", "hr"], "aliases": {}, "parentLocales": {} }, { "data": { "hsb": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} a {1}", "pair": "{0} a {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} a {1}", "pair": "{0} a {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} a {1}", "pair": "{0} a {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} a {1}", "pair": "{0} a {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} a {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["hsb"], "aliases": {}, "parentLocales": {} }, { "data": { "hu": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} és {1}", "pair": "{0} és {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} és {1}", "pair": "{0} és {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} és {1}", "pair": "{0} és {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} vagy {1}", "pair": "{0} vagy {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} vagy {1}", "pair": "{0} vagy {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} vagy {1}", "pair": "{0} vagy {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} és {1}", "pair": "{0} és {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} és {1}", "pair": "{0} és {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} és {1}", "pair": "{0} és {1}" } } } }, "availableLocales": ["hu"], "aliases": {}, "parentLocales": {} }, { "data": { "hy": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} և {1}", "pair": "{0} և {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} և {1}", "pair": "{0} և {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} կամ {1}", "pair": "{0} կամ {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} կամ {1}", "pair": "{0} կամ {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} կամ {1}", "pair": "{0} կամ {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} և {1}", "pair": "{0} և {1}" }, "short": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} և {1}", "pair": "{0} և {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["hy"], "aliases": {}, "parentLocales": {} }, { "data": { "ia": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} o {1}", "pair": "{0} o {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} o {1}", "pair": "{0} o {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} o {1}", "pair": "{0} o {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["ia"], "aliases": {}, "parentLocales": {} }, { "data": { "id": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, dan {1}", "pair": "{0} dan {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, dan {1}", "pair": "{0} dan {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, atau {1}", "pair": "{0} atau {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, atau {1}", "pair": "{0} atau {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, atau {1}", "pair": "{0} atau {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["id"], "aliases": {}, "parentLocales": {} }, { "data": { "ig": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, na {1}", "pair": "{0} na {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, na {1}", "pair": "{0} na {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, na {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, na {1}", "pair": "{0} na {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, na {1}", "pair": "{0} na {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, na {1}", "pair": "{0} na {1}" } } } }, "availableLocales": ["ig"], "aliases": {}, "parentLocales": {} }, { "data": { "ii": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["ii"], "aliases": {}, "parentLocales": {} }, { "data": { "is": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} eða {1}", "pair": "{0} eða {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} eða {1}", "pair": "{0} eða {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} eða {1}", "pair": "{0} eða {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} og {1}", "pair": "{0} og {1}" } } } }, "availableLocales": ["is"], "aliases": {}, "parentLocales": {} }, { "data": { "it": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} o {1}", "pair": "{0} o {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} o {1}", "pair": "{0} o {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} o {1}", "pair": "{0} o {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["it-CH", "it-SM", "it-VA", "it"], "aliases": {}, "parentLocales": {} }, { "data": { "ja": { "conjunction": { "long": { "start": "{0}、{1}", "middle": "{0}、{1}", "end": "{0}、{1}", "pair": "{0}、{1}" }, "short": { "start": "{0}、{1}", "middle": "{0}、{1}", "end": "{0}、{1}", "pair": "{0}、{1}" }, "narrow": { "start": "{0}、{1}", "middle": "{0}、{1}", "end": "{0}、{1}", "pair": "{0}、{1}" } }, "disjunction": { "long": { "start": "{0}、{1}", "middle": "{0}、{1}", "end": "{0}、または{1}", "pair": "{0}または{1}" }, "short": { "start": "{0}、{1}", "middle": "{0}、{1}", "end": "{0}、または{1}", "pair": "{0}または{1}" }, "narrow": { "start": "{0}、{1}", "middle": "{0}、{1}", "end": "{0}、または{1}", "pair": "{0}または{1}" } }, "unit": { "long": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "short": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "narrow": { "start": "{0}{1}", "middle": "{0}{1}", "end": "{0}{1}", "pair": "{0}{1}" } } } }, "availableLocales": ["ja"], "aliases": {}, "parentLocales": {} }, { "data": { "jgo": { "conjunction": { "long": { "start": "{0}, ŋ́gɛ {1}", "middle": "{0}, ŋ́gɛ {1}", "end": "{0}, ḿbɛn ŋ́gɛ {1}", "pair": "{0} pɔp {1}" }, "short": { "start": "{0}, ŋ́gɛ {1}", "middle": "{0}, ŋ́gɛ {1}", "end": "{0}, ḿbɛn ŋ́gɛ {1}", "pair": "{0} pɔp {1}" }, "narrow": { "start": "{0}, ŋ́gɛ {1}", "middle": "{0}, ŋ́gɛ {1}", "end": "{0}, ḿbɛn ŋ́gɛ {1}", "pair": "{0} pɔp {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, ŋ́gɛ {1}", "middle": "{0}, ŋ́gɛ {1}", "end": "{0}, ḿbɛn ŋ́gɛ {1}", "pair": "{0} pɔp {1}" }, "short": { "start": "{0}, ŋ́gɛ {1}", "middle": "{0}, ŋ́gɛ {1}", "end": "{0}, ḿbɛn ŋ́gɛ {1}", "pair": "{0} pɔp {1}" }, "narrow": { "start": "{0}, ŋ́gɛ {1}", "middle": "{0}, ŋ́gɛ {1}", "end": "{0}, ḿbɛn ŋ́gɛ {1}", "pair": "{0} pɔp {1}" } } } }, "availableLocales": ["jgo"], "aliases": {}, "parentLocales": {} }, { "data": { "jmc": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["jmc"], "aliases": {}, "parentLocales": {} }, { "data": { "jv": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, lan {1}", "pair": "{0} lan {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, lan {1}", "pair": "{0} lan {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, lan {1}", "pair": "{0} lan {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, utowo {1}", "pair": "{0} utowo {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, utowo {1}", "pair": "{0} utowo {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, utowo {1}", "pair": "{0} utowo {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["jv"], "aliases": {}, "parentLocales": {} }, { "data": { "ka": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} და {1}", "pair": "{0} და {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} და {1}", "pair": "{0} და {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} და {1}", "pair": "{0} და {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ან {1}", "pair": "{0} ან {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ან {1}", "pair": "{0} ან {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ან {1}", "pair": "{0} ან {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["ka"], "aliases": {}, "parentLocales": {} }, { "data": { "kab": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["kab"], "aliases": {}, "parentLocales": {} }, { "data": { "kam": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["kam"], "aliases": {}, "parentLocales": {} }, { "data": { "kde": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["kde"], "aliases": {}, "parentLocales": {} }, { "data": { "kea": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} o {1}", "pair": "{0} o {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} o {1}", "pair": "{0} o {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} o {1}", "pair": "{0} o {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["kea"], "aliases": {}, "parentLocales": {} }, { "data": { "khq": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["khq"], "aliases": {}, "parentLocales": {} }, { "data": { "ki": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["ki"], "aliases": {}, "parentLocales": {} }, { "data": { "kk": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0} және {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0} және {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0} және {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, не болмаса {1}", "pair": "{0} не {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, не болмаса {1}", "pair": "{0} не {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, не болмаса {1}", "pair": "{0} не {1}" } }, "unit": { "long": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "short": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["kk"], "aliases": { "kk-Cyrl-KZ": "kk-KZ" }, "parentLocales": {} }, { "data": { "kkj": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["kkj"], "aliases": {}, "parentLocales": {} }, { "data": { "kl": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["kl"], "aliases": {}, "parentLocales": {} }, { "data": { "kln": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["kln"], "aliases": {}, "parentLocales": {} }, { "data": { "km": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} និង {1}", "pair": "{0} និង​{1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} និង {1}", "pair": "{0} និង​{1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} និង {1}", "pair": "{0} និង​{1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ឬ {1}", "pair": "{0} ឬ {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ឬ {1}", "pair": "{0} ឬ {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ឬ {1}", "pair": "{0} ឬ {1}" } }, "unit": { "long": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "short": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["km"], "aliases": {}, "parentLocales": {} }, { "data": { "kn": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, ಮತ್ತು {1}", "pair": "{0} ಮತ್ತು {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, ಮತ್ತು {1}", "pair": "{0} ಮತ್ತು {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, ಅಥವಾ {1}", "pair": "{0} ಅಥವಾ {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, ಅಥವಾ {1}", "pair": "{0} ಅಥವಾ {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, ಅಥವಾ {1}", "pair": "{0} ಅಥವಾ {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["kn"], "aliases": {}, "parentLocales": {} }, { "data": { "ko": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} 및 {1}", "pair": "{0} 및 {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} 및 {1}", "pair": "{0} 및 {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} 및 {1}", "pair": "{0} 및 {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} 또는 {1}", "pair": "{0} 또는 {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} 또는 {1}", "pair": "{0} 또는 {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} 또는 {1}", "pair": "{0} 또는 {1}" } }, "unit": { "long": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "short": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["ko-KP", "ko"], "aliases": {}, "parentLocales": {} }, { "data": { "kok": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0} & {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0} & {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, वा {1}", "pair": "{0} वा {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, वा {1}", "pair": "{0} वा {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, वा {1}", "pair": "{0} वा {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["kok"], "aliases": {}, "parentLocales": {} }, { "data": { "ks": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["ks"], "aliases": { "ks-Arab-IN": "ks-IN" }, "parentLocales": {} }, { "data": { "ksb": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["ksb"], "aliases": {}, "parentLocales": {} }, { "data": { "ksf": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["ksf"], "aliases": {}, "parentLocales": {} }, { "data": { "ksh": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} un {1}", "pair": "{0} un {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} un {1}", "pair": "{0} un {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} un {1}", "pair": "{0} un {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["ksh"], "aliases": {}, "parentLocales": {} }, { "data": { "ku": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} û {1}", "pair": "{0} û {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} û {1}", "pair": "{0} û {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} û {1}", "pair": "{0} û {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} an {1}", "pair": "{0} an {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} an {1}", "pair": "{0} an {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} an {1}", "pair": "{0} an {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} û {1}", "pair": "{0} û {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} û {1}", "pair": "{0} û {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["ku"], "aliases": {}, "parentLocales": {} }, { "data": { "kw": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["kw"], "aliases": {}, "parentLocales": {} }, { "data": { "ky": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} жана {1}", "pair": "{0} жана {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} жана {1}", "pair": "{0} жана {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} жана {1}", "pair": "{0} жана {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, же {1}", "pair": "{0} же {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, же {1}", "pair": "{0} же {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, же {1}", "pair": "{0} же {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["ky"], "aliases": { "ky-Cyrl-KG": "ky-KG" }, "parentLocales": {} }, { "data": { "lag": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["lag"], "aliases": {}, "parentLocales": {} }, { "data": { "lb": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} a(n) {1}", "pair": "{0} a(n) {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} a(n) {1}", "pair": "{0} a(n) {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} a(n) {1}", "pair": "{0} a(n) {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["lb"], "aliases": {}, "parentLocales": {} }, { "data": { "lg": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["lg"], "aliases": {}, "parentLocales": {} }, { "data": { "lkt": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["lkt"], "aliases": {}, "parentLocales": {} }, { "data": { "ln": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["ln-AO", "ln-CF", "ln-CG", "ln"], "aliases": {}, "parentLocales": {} }, { "data": { "lo": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0} ແລະ {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0} ແລະ {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0} ແລະ {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ຫຼື {1}", "pair": "{0} ຫຼື {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ຫຼື {1}", "pair": "{0} ຫຼື {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ຫຼື {1}", "pair": "{0} ຫຼື {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["lo"], "aliases": {}, "parentLocales": {} }, { "data": { "lrc": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["lrc-IQ", "lrc"], "aliases": {}, "parentLocales": {} }, { "data": { "lt": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ir {1}", "pair": "{0} ir {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ir {1}", "pair": "{0} ir {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ir {1}", "pair": "{0} ir {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ar {1}", "pair": "{0} ar {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ar {1}", "pair": "{0} ar {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ar {1}", "pair": "{0} ar {1}" } }, "unit": { "long": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} ir {1}", "pair": "{0} ir {1}" }, "short": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["lt"], "aliases": {}, "parentLocales": {} }, { "data": { "lu": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["lu"], "aliases": {}, "parentLocales": {} }, { "data": { "luo": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["luo"], "aliases": {}, "parentLocales": {} }, { "data": { "luy": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["luy"], "aliases": {}, "parentLocales": {} }, { "data": { "lv": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} un {1}", "pair": "{0} un {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} un {1}", "pair": "{0} un {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} un {1}", "pair": "{0} un {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} vai {1}", "pair": "{0} vai {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} vai {1}", "pair": "{0} vai {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} vai {1}", "pair": "{0} vai {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} un {1}", "pair": "{0} un {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} un {1}", "pair": "{0} un {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["lv"], "aliases": {}, "parentLocales": {} }, { "data": { "mas": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["mas-TZ", "mas"], "aliases": {}, "parentLocales": {} }, { "data": { "mer": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["mer"], "aliases": {}, "parentLocales": {} }, { "data": { "mfe": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["mfe"], "aliases": {}, "parentLocales": {} }, { "data": { "mg": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["mg"], "aliases": {}, "parentLocales": {} }, { "data": { "mgh": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["mgh"], "aliases": {}, "parentLocales": {} }, { "data": { "mgo": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["mgo"], "aliases": {}, "parentLocales": {} }, { "data": { "mi": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["mi"], "aliases": {}, "parentLocales": {} }, { "data": { "mk": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} или {1}", "pair": "{0} или {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} или {1}", "pair": "{0} или {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} или {1}", "pair": "{0} или {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" } } } }, "availableLocales": ["mk"], "aliases": {}, "parentLocales": {} }, { "data": { "ml": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1} എന്നിവ", "pair": "{0} കൂടാതെ {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1} എന്നിവ", "pair": "{0} കൂടാതെ {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1} എന്നിവ", "pair": "{0} കൂടാതെ {1}" } }, "disjunction": { "long": { "start": "{0}, {1} എന്നിവ", "middle": "{0}, {1} എന്നിവ", "end": "{0}, അല്ലെങ്കിൽ {1}", "pair": "{0} അല്ലെങ്കിൽ {1}" }, "short": { "start": "{0}, {1} എന്നിവ", "middle": "{0}, {1} എന്നിവ", "end": "{0}, അല്ലെങ്കിൽ {1}", "pair": "{0} അല്ലെങ്കിൽ {1}" }, "narrow": { "start": "{0}, {1} എന്നിവ", "middle": "{0}, {1} എന്നിവ", "end": "{0}, അല്ലെങ്കിൽ {1}", "pair": "{0} അല്ലെങ്കിൽ {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0} കൂടാതെ {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0} കൂടാതെ {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["ml"], "aliases": {}, "parentLocales": {} }, { "data": { "mn": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1} зэргийн аль нэг", "pair": "{0} эсвэл {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1} зэргийн аль нэг", "pair": "{0} эсвэл {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1} зэргийн аль нэг", "pair": "{0} эсвэл {1}" } }, "unit": { "long": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "short": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["mn"], "aliases": { "mn-Cyrl-MN": "mn-MN" }, "parentLocales": {} }, { "data": { "mr": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} आणि {1}", "pair": "{0} आणि {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} आणि {1}", "pair": "{0} आणि {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} आणि {1}", "pair": "{0} आणि {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, किंवा {1}", "pair": "{0} किंवा {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, किंवा {1}", "pair": "{0} किंवा {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, किंवा {1}", "pair": "{0} किंवा {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["mr"], "aliases": {}, "parentLocales": {} }, { "data": { "ms": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} dan {1}", "pair": "{0} dan {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} dan {1}", "pair": "{0} dan {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, atau {1}", "pair": "{0} atau {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, atau {1}", "pair": "{0} atau {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, atau {1}", "pair": "{0} atau {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} dan {1}", "pair": "{0} dan {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["ms-BN", "ms-SG", "ms"], "aliases": { "ms-Latn-BN": "ms-BN", "ms-Latn-MY": "ms-MY", "ms-Latn-SG": "ms-SG" }, "parentLocales": {} }, { "data": { "mt": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, u {1}", "pair": "{0} u {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, u {1}", "pair": "{0} u {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, u {1}", "pair": "{0} u {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, u {1}", "pair": "{0} u {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, u {1}", "pair": "{0} u {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["mt"], "aliases": {}, "parentLocales": {} }, { "data": { "mua": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["mua"], "aliases": {}, "parentLocales": {} }, { "data": { "my": { "conjunction": { "long": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0}နှင့် {1}", "pair": "{0}နှင့် {1}" }, "short": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0}နှင့် {1}", "pair": "{0}နှင့် {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0}နှင့် {1}", "pair": "{0}နှင့် {1}" } }, "disjunction": { "long": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} သို့မဟုတ် {1}", "pair": "{0} သို့မဟုတ် {1}" }, "short": { "start": "{0} ၊ {1}", "middle": "{0} {1}", "end": "{0} သို့မဟုတ် {1}", "pair": "{0} သို့မဟုတ် {1}" }, "narrow": { "start": "{0} ၊ {1}", "middle": "{0} {1}", "end": "{0} သို့မဟုတ် {1}", "pair": "{0} သို့မဟုတ် {1}" } }, "unit": { "long": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0}နှင့် {1}", "pair": "{0}နှင့်{1}" }, "short": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0}နှင့် {1}", "pair": "{0}နှင့် {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0}နှင့် {1}", "pair": "{0}နှင့် {1}" } } } }, "availableLocales": ["my"], "aliases": {}, "parentLocales": {} }, { "data": { "mzn": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["mzn"], "aliases": {}, "parentLocales": {} }, { "data": { "naq": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["naq"], "aliases": {}, "parentLocales": {} }, { "data": { "nb": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} eller {1}", "pair": "{0} eller {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} eller {1}", "pair": "{0} eller {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} eller {1}", "pair": "{0} eller {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["nb-SJ", "nb"], "aliases": {}, "parentLocales": {} }, { "data": { "nd": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["nd"], "aliases": {}, "parentLocales": {} }, { "data": { "nds": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["nds-NL", "nds"], "aliases": {}, "parentLocales": {} }, { "data": { "ne": { "conjunction": { "long": { "start": "{0},{1}", "middle": "{0}, {1}", "end": "{0} र {1}", "pair": "{0} र {1}" }, "short": { "start": "{0},{1}", "middle": "{0}, {1}", "end": "{0} र {1}", "pair": "{0} र {1}" }, "narrow": { "start": "{0},{1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, वा {1}", "pair": "{0} वा {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, वा {1}", "pair": "{0} वा {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, वा {1}", "pair": "{0} वा {1}" } }, "unit": { "long": { "start": "{0},{1}", "middle": "{0}, {1}", "end": "{0},{1}", "pair": "{0},{1}" }, "short": { "start": "{0},{1}", "middle": "{0}, {1}", "end": "{0},{1}", "pair": "{0} {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0}{1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["ne-IN", "ne"], "aliases": {}, "parentLocales": {} }, { "data": { "nl": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} en {1}", "pair": "{0} en {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} en {1}", "pair": "{0} en {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} en {1}", "pair": "{0} en {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} of {1}", "pair": "{0} of {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} of {1}", "pair": "{0} of {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} of {1}", "pair": "{0} of {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} en {1}", "pair": "{0} en {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["nl-AW", "nl-BE", "nl-BQ", "nl-CW", "nl-SR", "nl-SX", "nl"], "aliases": {}, "parentLocales": {} }, { "data": { "nmg": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["nmg"], "aliases": {}, "parentLocales": {} }, { "data": { "nn": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} og {1}", "pair": "{0} og {1}" } } } }, "availableLocales": ["nn"], "aliases": {}, "parentLocales": {} }, { "data": { "nnh": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["nnh"], "aliases": {}, "parentLocales": {} }, { "data": { "nus": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["nus"], "aliases": {}, "parentLocales": {} }, { "data": { "nyn": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["nyn"], "aliases": {}, "parentLocales": {} }, { "data": { "om": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["om-KE", "om"], "aliases": {}, "parentLocales": {} }, { "data": { "or": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, ଓ {1}", "pair": "{0} ଓ {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, ଓ {1}", "pair": "{0} ଓ {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, ଓ {1}", "pair": "{0} ଓ {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} କିମ୍ବା {1}", "pair": "{0} କିମ୍ବା {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} କିମ୍ବା {1}", "pair": "{0} କିମ୍ବା {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} କିମ୍ବା {1}", "pair": "{0} କିମ୍ବା {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["or"], "aliases": {}, "parentLocales": {} }, { "data": { "os": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ӕмӕ {1}", "pair": "{0} ӕмӕ {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ӕмӕ {1}", "pair": "{0} ӕмӕ {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ӕмӕ {1}", "pair": "{0} ӕмӕ {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ӕмӕ {1}", "pair": "{0} ӕмӕ {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ӕмӕ {1}", "pair": "{0} ӕмӕ {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ӕмӕ {1}", "pair": "{0} ӕмӕ {1}" } } } }, "availableLocales": ["os-RU", "os"], "aliases": {}, "parentLocales": {} }, { "data": { "pa-Arab": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } }, "pa": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ਅਤੇ {1}", "pair": "{0} ਅਤੇ {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ਅਤੇ {1}", "pair": "{0} ਅਤੇ {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} ਜਾਂ {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, ਜਾਂ {1}", "pair": "{0} ਜਾਂ {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, ਜਾਂ {1}", "pair": "{0} ਜਾਂ {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["pa-Arab", "pa-Guru", "pa"], "aliases": { "pa-IN": "pa-Guru-IN", "pa-PK": "pa-Arab-PK" }, "parentLocales": {} }, { "data": { "pl": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} lub {1}", "pair": "{0} lub {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} lub {1}", "pair": "{0} lub {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} lub {1}", "pair": "{0} lub {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" } } } }, "availableLocales": ["pl"], "aliases": {}, "parentLocales": {} }, { "data": { "prg": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["prg"], "aliases": {}, "parentLocales": {} }, { "data": { "ps": { "conjunction": { "long": { "start": "{0}، {1}", "middle": "{0}، {1}", "end": "{0}، او {1}", "pair": "{0} او {1}" }, "short": { "start": "{0}، {1}", "middle": "{0}، {1}", "end": "{0}، او {1}", "pair": "{0} او {1}" }, "narrow": { "start": "{0}، {1}", "middle": "{0}، {1}", "end": "{0}، او {1}", "pair": "{0}، {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0} او {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0} و {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["ps-PK", "ps"], "aliases": {}, "parentLocales": {} }, { "data": { "pt-PT": { "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" } } }, "pt": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ou {1}", "pair": "{0} ou {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ou {1}", "pair": "{0} ou {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ou {1}", "pair": "{0} ou {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["pt-AO", "pt-CH", "pt-CV", "pt-GQ", "pt-GW", "pt-LU", "pt-MO", "pt-MZ", "pt-PT", "pt-ST", "pt-TL", "pt"], "aliases": {}, "parentLocales": { "pt-AO": "pt-PT", "pt-CH": "pt-PT", "pt-CV": "pt-PT", "pt-FR": "pt-PT", "pt-GQ": "pt-PT", "pt-GW": "pt-PT", "pt-LU": "pt-PT", "pt-MO": "pt-PT", "pt-MZ": "pt-PT", "pt-ST": "pt-PT", "pt-TL": "pt-PT" } }, { "data": { "qu": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, utaq {1}", "pair": "{0} utaq {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, utaq {1}", "pair": "{0} utaq {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, utaq {1}", "pair": "{0} utaq {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["qu-BO", "qu-EC", "qu"], "aliases": {}, "parentLocales": {} }, { "data": { "rm": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["rm"], "aliases": {}, "parentLocales": {} }, { "data": { "rn": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["rn"], "aliases": {}, "parentLocales": {} }, { "data": { "ro": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} și {1}", "pair": "{0} și {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} și {1}", "pair": "{0} și {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} sau {1}", "pair": "{0} sau {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} sau {1}", "pair": "{0} sau {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} sau {1}", "pair": "{0} sau {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0} și {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["ro-MD", "ro"], "aliases": {}, "parentLocales": {} }, { "data": { "rof": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["rof"], "aliases": {}, "parentLocales": {} }, { "data": { "root": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["root"], "aliases": {}, "parentLocales": {} }, { "data": { "ru": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} или {1}", "pair": "{0} или {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} или {1}", "pair": "{0} или {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} или {1}", "pair": "{0} или {1}" } }, "unit": { "long": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "short": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["ru-BY", "ru-KG", "ru-KZ", "ru-MD", "ru-UA", "ru"], "aliases": {}, "parentLocales": {} }, { "data": { "rw": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["rw"], "aliases": {}, "parentLocales": {} }, { "data": { "rwk": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["rwk"], "aliases": {}, "parentLocales": {} }, { "data": { "sah": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} уонна {1}", "pair": "{0} уонна {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} уонна {1}", "pair": "{0} уонна {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} уонна {1}", "pair": "{0} уонна {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} уонна {1}", "pair": "{0} уонна {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} уонна {1}", "pair": "{0} уонна {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} уонна {1}", "pair": "{0} уонна {1}" } } } }, "availableLocales": ["sah"], "aliases": {}, "parentLocales": {} }, { "data": { "saq": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["saq"], "aliases": {}, "parentLocales": {} }, { "data": { "sbp": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["sbp"], "aliases": {}, "parentLocales": {} }, { "data": { "sd": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}، ۽ {1}", "pair": "{0} ۽ {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}، ۽ {1}", "pair": "{0} ۽ {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, يا {1}", "pair": "{0} يا {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, يا {1}", "pair": "{0} يا {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, يا {1}", "pair": "{0} يا {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["sd"], "aliases": {}, "parentLocales": {} }, { "data": { "se": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ja {1}", "pair": "{0} ja {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ja {1}", "pair": "{0} ja {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ja {1}", "pair": "{0} ja {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["se-FI", "se-SE", "se"], "aliases": {}, "parentLocales": {} }, { "data": { "seh": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["seh"], "aliases": {}, "parentLocales": {} }, { "data": { "ses": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["ses"], "aliases": {}, "parentLocales": {} }, { "data": { "sg": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["sg"], "aliases": {}, "parentLocales": {} }, { "data": { "shi": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["shi-Latn", "shi-Tfng", "shi"], "aliases": { "shi-MA": "shi-Tfng-MA" }, "parentLocales": {} }, { "data": { "si": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, සහ {1}", "pair": "{0} සහ {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, සහ {1}", "pair": "{0} සහ {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, සහ {1}", "pair": "{0} සහ {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, හෝ {1}", "pair": "{0} හෝ {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, හෝ {1}", "pair": "{0} හෝ {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, හෝ {1}", "pair": "{0} හෝ {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, සහ {1}", "pair": "{0} සහ {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, සහ {1}", "pair": "{0} සහ {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, සහ {1}", "pair": "{0} සහ {1}" } } } }, "availableLocales": ["si"], "aliases": {}, "parentLocales": {} }, { "data": { "sk": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} a {1}", "pair": "{0} a {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} a {1}", "pair": "{0} a {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} a {1}", "pair": "{0} a {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} alebo {1}", "pair": "{0} alebo {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} alebo {1}", "pair": "{0} alebo {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} alebo {1}", "pair": "{0} alebo {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["sk"], "aliases": {}, "parentLocales": {} }, { "data": { "sl": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} in {1}", "pair": "{0} in {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} in {1}", "pair": "{0} in {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} in {1}", "pair": "{0} in {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ali {1}", "pair": "{0} ali {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ali {1}", "pair": "{0} ali {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ali {1}", "pair": "{0} ali {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} in {1}", "pair": "{0} in {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} in {1}", "pair": "{0} in {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} in {1}", "pair": "{0} in {1}" } } } }, "availableLocales": ["sl"], "aliases": {}, "parentLocales": {} }, { "data": { "smn": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["smn"], "aliases": {}, "parentLocales": {} }, { "data": { "sn": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["sn"], "aliases": {}, "parentLocales": {} }, { "data": { "so": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} iyo {1}", "pair": "{0} iyo {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} & {1}", "pair": "{0} & {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ama {1}", "pair": "{0} ama {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ama {1}", "pair": "{0} ama {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ama {1}", "pair": "{0} ama {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0} iyo {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["so-DJ", "so-ET", "so-KE", "so"], "aliases": {}, "parentLocales": {} }, { "data": { "sq": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} dhe {1}", "pair": "{0} dhe {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} dhe {1}", "pair": "{0} dhe {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} dhe {1}", "pair": "{0} dhe {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ose {1}", "pair": "{0} ose {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ose {1}", "pair": "{0} ose {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ose {1}", "pair": "{0} ose {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} e {1}", "pair": "{0} e {1}" } } } }, "availableLocales": ["sq-MK", "sq-XK", "sq"], "aliases": {}, "parentLocales": {} }, { "data": { "sr-Latn": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ili {1}", "pair": "{0} ili {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ili {1}", "pair": "{0} ili {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ili {1}", "pair": "{0} ili {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} i {1}", "pair": "{0} i {1}" } } }, "sr": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} или {1}", "pair": "{0} или {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} или {1}", "pair": "{0} или {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} или {1}", "pair": "{0} или {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} и {1}", "pair": "{0} и {1}" } } } }, "availableLocales": ["sr-Cyrl-BA", "sr-Cyrl-ME", "sr-Cyrl-XK", "sr-Cyrl", "sr-Latn-BA", "sr-Latn-ME", "sr-Latn-XK", "sr-Latn", "sr"], "aliases": { "sr-BA": "sr-Cyrl-BA", "sr-ME": "sr-Latn-ME", "sr-RS": "sr-Cyrl-RS", "sr-XK": "sr-Cyrl-XK" }, "parentLocales": {} }, { "data": { "sv": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} och {1}", "pair": "{0} och {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} och {1}", "pair": "{0} och {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} och {1}", "pair": "{0} och {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} eller {1}", "pair": "{0} eller {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} eller {1}", "pair": "{0} eller {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} eller {1}", "pair": "{0} eller {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["sv-AX", "sv-FI", "sv"], "aliases": {}, "parentLocales": {} }, { "data": { "sw": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} na {1}", "pair": "{0} na {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} na {1}", "pair": "{0} na {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} na {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} au {1}", "pair": "{0} au {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, au {1}", "pair": "{0} au {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, au {1}", "pair": "{0} au {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} na {1}", "pair": "{0} na {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} na {1}", "pair": "{0} na {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} na {1}", "pair": "{0} na {1}" } } } }, "availableLocales": ["sw-CD", "sw-KE", "sw-UG", "sw"], "aliases": {}, "parentLocales": {} }, { "data": { "ta": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} மற்றும் {1}", "pair": "{0} மற்றும் {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} மற்றும் {1}", "pair": "{0} மற்றும் {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} மற்றும் {1}", "pair": "{0} மற்றும் {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} அல்லது {1}", "pair": "{0} அல்லது {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} அல்லது {1}", "pair": "{0} அல்லது {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} அல்லது {1}", "pair": "{0} அல்லது {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} மற்றும் {1}", "pair": "{0} மற்றும் {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} மற்றும் {1}", "pair": "{0} மற்றும் {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["ta-LK", "ta-MY", "ta-SG", "ta"], "aliases": {}, "parentLocales": {} }, { "data": { "te": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} మరియు {1}", "pair": "{0} మరియు {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} మరియు {1}", "pair": "{0} మరియు {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} లేదా {1}", "pair": "{0} లేదా {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} లేదా {1}", "pair": "{0} లేదా {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} లేదా {1}", "pair": "{0} లేదా {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["te"], "aliases": {}, "parentLocales": {} }, { "data": { "teo": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["teo-KE", "teo"], "aliases": {}, "parentLocales": {} }, { "data": { "tg": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["tg"], "aliases": {}, "parentLocales": {} }, { "data": { "th": { "conjunction": { "long": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} และ{1}", "pair": "{0}และ{1}" }, "short": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} และ{1}", "pair": "{0}และ{1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} และ{1}", "pair": "{0}และ{1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} หรือ {1}", "pair": "{0} หรือ {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} หรือ {1}", "pair": "{0}หรือ{1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} หรือ {1}", "pair": "{0}หรือ{1}" } }, "unit": { "long": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} และ {1}", "pair": "{0} และ {1}" }, "short": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} และ {1}", "pair": "{0} {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["th"], "aliases": {}, "parentLocales": {} }, { "data": { "ti": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["ti-ER", "ti"], "aliases": {}, "parentLocales": {} }, { "data": { "tk": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} we {1}", "pair": "{0} we {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} we {1}", "pair": "{0} we {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ýa-da {1}", "pair": "{0} ýa-da {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ýa-da {1}", "pair": "{0} ýa-da {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ýa-da {1}", "pair": "{0} ýa-da {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["tk"], "aliases": {}, "parentLocales": {} }, { "data": { "to": { "conjunction": { "long": { "start": "{0} mo {1}", "middle": "{0} mo {1}", "end": "{0} mo {1}", "pair": "{0} mo {1}" }, "short": { "start": "{0} mo {1}", "middle": "{0} mo {1}", "end": "{0} mo {1}", "pair": "{0} mo {1}" }, "narrow": { "start": "{0} mo {1}", "middle": "{0} mo {1}", "end": "{0} mo {1}", "pair": "{0} mo {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, pē {1}", "pair": "{0} pē {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, pē {1}", "pair": "{0} pē {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, pē {1}", "pair": "{0} pē {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} mo e {1}", "pair": "{0} mo e {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} mo e {1}", "pair": "{0} mo e {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} mo e {1}", "pair": "{0} mo e {1}" } } } }, "availableLocales": ["to"], "aliases": {}, "parentLocales": {} }, { "data": { "tr": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ve {1}", "pair": "{0} ve {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} ve {1}", "pair": "{0} ve {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} veya {1}", "pair": "{0} veya {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} veya {1}", "pair": "{0} veya {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} veya {1}", "pair": "{0} veya {1}" } }, "unit": { "long": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "short": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["tr-CY", "tr"], "aliases": {}, "parentLocales": {} }, { "data": { "tt": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} һәм {1}", "pair": "{0} һәм {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} һәм {1}", "pair": "{0} һәм {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} һәм {1}", "pair": "{0} һәм {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} һәм {1}", "pair": "{0} һәм {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} һәм {1}", "pair": "{0} һәм {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} һәм {1}", "pair": "{0} һәм {1}" } } } }, "availableLocales": ["tt"], "aliases": {}, "parentLocales": {} }, { "data": { "twq": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["twq"], "aliases": {}, "parentLocales": {} }, { "data": { "tzm": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["tzm"], "aliases": { "tzm-Latn-MA": "tzm-MA" }, "parentLocales": {} }, { "data": { "ug": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, and {1}", "pair": "{0} and {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, and {1}", "pair": "{0} and {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, and {1}", "pair": "{0} and {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, and {1}", "pair": "{0} and {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, and {1}", "pair": "{0} and {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, and {1}", "pair": "{0} and {1}" } } } }, "availableLocales": ["ug"], "aliases": { "ug-Arab-CN": "ug-CN" }, "parentLocales": {} }, { "data": { "uk": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} і {1}", "pair": "{0} і {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} і {1}", "pair": "{0} і {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} або {1}", "pair": "{0} або {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} або {1}", "pair": "{0} або {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} або {1}", "pair": "{0} або {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} і {1}", "pair": "{0} і {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} і {1}", "pair": "{0} і {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} і {1}", "pair": "{0} і {1}" } } } }, "availableLocales": ["uk"], "aliases": {}, "parentLocales": {} }, { "data": { "ur-IN": { "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{1}، {0}", "pair": "{1}، {0}" }, "short": { "start": "{0}، {1}", "middle": "{0}، {1}", "end": "{1}، {0}", "pair": "{1}، {0}" }, "narrow": { "start": "{1} {0}", "middle": "{1} {0}", "end": "{1} {0}", "pair": "{1} {0}" } } }, "ur": { "conjunction": { "long": { "start": "{0}، {1}", "middle": "{0}، {1}", "end": "{0}، اور {1}", "pair": "{0} اور {1}" }, "short": { "start": "{0}، {1}", "middle": "{0}، {1}", "end": "{0}، اور {1}", "pair": "{0} اور {1}" }, "narrow": { "start": "{0}، {1}", "middle": "{0}، {1}", "end": "{0}، {1}", "pair": "{0}، {1}" } }, "disjunction": { "long": { "start": "{0}، {1}", "middle": "{0}، {1}", "end": "{0}، یا {1}", "pair": "{0} یا {1}" }, "short": { "start": "{0}، {1}", "middle": "{0}، {1}", "end": "{0}، یا {1}", "pair": "{0} یا {1}" }, "narrow": { "start": "{0}، {1}", "middle": "{0}، {1}", "end": "{0}، یا {1}", "pair": "{0} یا {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}، اور {1}", "pair": "{0}، {1}" }, "short": { "start": "{0}، {1}", "middle": "{0}، {1}", "end": "{0}، اور {1}", "pair": "{0} اور {1}" }, "narrow": { "start": "{0}، {1}", "middle": "{0}، {1}", "end": "{0}، اور {1}", "pair": "{0} اور {1}" } } } }, "availableLocales": ["ur-IN", "ur"], "aliases": {}, "parentLocales": {} }, { "data": { "uz-Arab": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } }, "uz-Cyrl": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } }, "uz": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} va {1}", "pair": "{0} va {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} va {1}", "pair": "{0} va {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} yoki {1}", "pair": "{0} yoki {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} yoki {1}", "pair": "{0} yoki {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} yoki {1}", "pair": "{0} yoki {1}" } }, "unit": { "long": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "short": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["uz-Arab", "uz-Cyrl", "uz-Latn", "uz"], "aliases": { "uz-AF": "uz-Arab-AF", "uz-UZ": "uz-Latn-UZ" }, "parentLocales": {} }, { "data": { "vai": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["vai-Latn", "vai-Vaii", "vai"], "aliases": { "vai-LR": "vai-Vaii-LR" }, "parentLocales": {} }, { "data": { "vi": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} và {1}", "pair": "{0} và {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} và {1}", "pair": "{0} và {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} hoặc {1}", "pair": "{0} hoặc {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} hoặc {1}", "pair": "{0} hoặc {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} hoặc {1}", "pair": "{0} hoặc {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} và {1}", "pair": "{0} và {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} và {1}", "pair": "{0} và {1}" }, "narrow": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" } } } }, "availableLocales": ["vi"], "aliases": {}, "parentLocales": {} }, { "data": { "vo": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["vo"], "aliases": {}, "parentLocales": {} }, { "data": { "vun": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["vun"], "aliases": {}, "parentLocales": {} }, { "data": { "wae": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} und {1}", "pair": "{0} und {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} und {1}", "pair": "{0} und {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} und {1}", "pair": "{0} und {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} und {1}", "pair": "{0} und {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} und {1}", "pair": "{0} und {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} und {1}", "pair": "{0} und {1}" } } } }, "availableLocales": ["wae"], "aliases": {}, "parentLocales": {} }, { "data": { "wo": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["wo"], "aliases": {}, "parentLocales": {} }, { "data": { "xh": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["xh"], "aliases": {}, "parentLocales": {} }, { "data": { "xog": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["xog"], "aliases": {}, "parentLocales": {} }, { "data": { "yav": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["yav"], "aliases": {}, "parentLocales": {} }, { "data": { "yi": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} און {1}", "pair": "{0} און {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} און {1}", "pair": "{0} און {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} און {1}", "pair": "{0} און {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} און {1}", "pair": "{0} און {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} און {1}", "pair": "{0} און {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} און {1}", "pair": "{0} און {1}" } } } }, "availableLocales": ["yi"], "aliases": {}, "parentLocales": {} }, { "data": { "yo": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["yo-BJ", "yo"], "aliases": {}, "parentLocales": {} }, { "data": { "yue": { "conjunction": { "long": { "start": "{0}、{1}", "middle": "{0}、{1}", "end": "{0}同{1}", "pair": "{0}同{1}" }, "short": { "start": "{0}、{1}", "middle": "{0}、{1}", "end": "{0}同{1}", "pair": "{0}同{1}" }, "narrow": { "start": "{0}、{1}", "middle": "{0}、{1}", "end": "{0}同{1}", "pair": "{0}同{1}" } }, "disjunction": { "long": { "start": "{0}、{1}", "middle": "{0}、{1}", "end": "{0} 或 {1}", "pair": "{0} 或 {1}" }, "short": { "start": "{0}、{1}", "middle": "{0}、{1}", "end": "{0} 或 {1}", "pair": "{0} 或 {1}" }, "narrow": { "start": "{0}、{1}", "middle": "{0}、{1}", "end": "{0} 或 {1}", "pair": "{0} 或 {1}" } }, "unit": { "long": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "short": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "narrow": { "start": "{0}{1}", "middle": "{0}{1}", "end": "{0}{1}", "pair": "{0}{1}" } } } }, "availableLocales": ["yue-Hans", "yue-Hant", "yue"], "aliases": { "yue-CN": "yue-Hans-CN", "yue-HK": "yue-Hant-HK" }, "parentLocales": {} }, { "data": { "zgh": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["zgh"], "aliases": {}, "parentLocales": {} }, { "data": { "zh-Hant-HK": { "conjunction": { "long": { "start": "{0}、{1}", "middle": "{0}、{1}", "end": "{0}及{1}", "pair": "{0}及{1}" }, "short": { "start": "{0}、{1}", "middle": "{0}、{1}", "end": "{0}及{1}", "pair": "{0}及{1}" }, "narrow": { "start": "{0}、{1}", "middle": "{0}、{1}", "end": "{0}及{1}", "pair": "{0}及{1}" } } }, "zh-Hant": { "unit": { "long": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "short": { "start": "{0} {1}", "middle": "{0} {1}", "end": "{0} {1}", "pair": "{0} {1}" }, "narrow": { "start": "{0}{1}", "middle": "{0}{1}", "end": "{0}{1}", "pair": "{0}{1}" } } }, "zh": { "conjunction": { "long": { "start": "{0}、{1}", "middle": "{0}、{1}", "end": "{0}和{1}", "pair": "{0}和{1}" }, "short": { "start": "{0}、{1}", "middle": "{0}、{1}", "end": "{0}和{1}", "pair": "{0}和{1}" }, "narrow": { "start": "{0}、{1}", "middle": "{0}、{1}", "end": "{0}和{1}", "pair": "{0}和{1}" } }, "disjunction": { "long": { "start": "{0}、{1}", "middle": "{0}、{1}", "end": "{0}或{1}", "pair": "{0}或{1}" }, "short": { "start": "{0}、{1}", "middle": "{0}、{1}", "end": "{0}或{1}", "pair": "{0}或{1}" }, "narrow": { "start": "{0}、{1}", "middle": "{0}、{1}", "end": "{0}或{1}", "pair": "{0}或{1}" } }, "unit": { "long": { "start": "{0}{1}", "middle": "{0}{1}", "end": "{0}{1}", "pair": "{0}{1}" }, "short": { "start": "{0}{1}", "middle": "{0}{1}", "end": "{0}{1}", "pair": "{0}{1}" }, "narrow": { "start": "{0}{1}", "middle": "{0}{1}", "end": "{0}{1}", "pair": "{0}{1}" } } } }, "availableLocales": ["zh-Hans-HK", "zh-Hans-MO", "zh-Hans-SG", "zh-Hans", "zh-Hant-HK", "zh-Hant-MO", "zh-Hant", "zh"], "aliases": { "zh-CN": "zh-Hans-CN", "zh-guoyu": "zh", "zh-hakka": "hak", "zh-HK": "zh-Hant-HK", "zh-min-nan": "nan", "zh-MO": "zh-Hant-MO", "zh-SG": "zh-Hans-SG", "zh-TW": "zh-Hant-TW", "zh-xiang": "hsn", "zh-min": "nan-x-zh-min" }, "parentLocales": { "zh-Hant-MO": "zh-Hant-HK" } }, { "data": { "zu": { "conjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, ne-{1}", "pair": "{0} ne-{1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, ne-{1}", "pair": "{0} ne-{1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" } }, "disjunction": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, or {1}", "pair": "{0} or {1}" } }, "unit": { "long": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "short": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0}, {1}", "pair": "{0}, {1}" }, "narrow": { "start": "{0}, {1}", "middle": "{0}, {1}", "end": "{0} {1}", "pair": "{0}, {1}" } } } }, "availableLocales": ["zu"], "aliases": {}, "parentLocales": {} });

    exports.default = ListFormat;

    Object.defineProperty(exports, '__esModule', { value: true });

})));
//# sourceMappingURL=intl-listformat-with-locales.js.map
